package processors

import (
	"context"
	"strconv"
	"sync"
	"time"

	"code.justin.tv/cb/kinesis_processor/adapters"
	"code.justin.tv/cb/kinesis_processor/clients"
	"code.justin.tv/cb/kinesis_processor/clients/kinesis_firehose"
	"code.justin.tv/cb/kinesis_processor/models"
	"code.justin.tv/cb/kinesis_processor/utils"
	log "github.com/sirupsen/logrus"
)

const (
	// ChatParallelWorkersCount - Number of go channels we will create
	ChatParallelWorkersCount = 200
	// ChatBufferSize - Buffer size
	ChatBufferSize = 50
)

// Chat interface consumed in app.go
type Chat interface {
	Before() error
	Flush() error
	ProcessEvent(event kinesis_firehose.KinesisEvent) error
	After() error
}

type chat struct {
	clients     *clients.Clients
	adapter     adapters.ChatAdapter
	buffer      []models.ChatEvent
	wg          sync.WaitGroup
	records     chan []models.ChatEvent
	deadCounter int
}

// NewChat creates a new processor
func NewChat(clients *clients.Clients) Chat {
	return &chat{
		clients: clients,
		adapter: adapters.NewChatAdapter(clients.Conf.Environment, clients.Conf.AWSRegion, clients.Conf.ChatActivityRedisHost),
	}
}

// Before - start worksers
func (p *chat) Before() error {
	p.buffer = []models.ChatEvent{}
	p.records = make(chan []models.ChatEvent, ChatParallelWorkersCount)

	for w := 0; w < ChatParallelWorkersCount; w++ {
		p.startWorker(w)
	}

	return nil
}

// startWorker - function called x amount of times for each go channel
func (p *chat) startWorker(number int) {
	p.wg.Add(1)

	go func(number int) {
		defer func() {
			p.wg.Done()

			if r := recover(); r != nil {
				log.Error("Failed worker: ", r)
			}

			time.Sleep(1 * time.Minute)
			p.startWorker(number)
		}()

		var err error
		for models := range p.records {
			err = p.adapter.BatchSaveChatEvents(context.TODO(), models)
			if err != nil {
				if IsDuplicateError(err) {
					continue
				}
				log.Error(err)
				continue
			}
		}
	}(number)
}

// Flush - cleans ouut the buffer
func (p *chat) Flush() error {
	tmp := make([]models.ChatEvent, len(p.buffer))
	copy(tmp, p.buffer)
	p.records <- tmp
	p.buffer = []models.ChatEvent{}

	return nil
}

// ProcessEvent
func (p *chat) ProcessEvent(event kinesis_firehose.KinesisEvent) error {
	channelID, err := strconv.ParseInt(event.Fields["channel_id"], 10, 64)
	if err != nil {
		return nil
	}

	userID, err := strconv.ParseInt(event.Fields["user_id"], 10, 64)
	if err != nil {
		return nil
	}

	var eventTime string
	if event.Fields["time_utc"] != "" {
		eventTime = event.Fields["time_utc"]
	} else {
		result, err := utils.ConvertPSTtoUTC(event.Fields["time"])
		if err != nil {
			return nil
		}
		eventTime = *result
	}

	eventTimeTime, err := time.Parse(utils.DbTimeFormat, eventTime)
	if err != nil {
		return err
	}

	obj := models.ChatEvent{
		ChannelID: channelID,
		UserID:    userID,
		Time:      eventTimeTime,
	}

	p.buffer = append(p.buffer, obj)
	if len(p.buffer) >= ChatBufferSize {
		err = p.Flush()
		if err != nil {
			return err
		}
	}

	return nil
}

func (p *chat) After() error {
	err := p.Flush()
	if err != nil {
		return err
	}

	close(p.records)
	p.wg.Wait()
	return nil
}
