package processors

import (
	"context"
	"errors"
	"strconv"
	"sync"
	"time"

	"code.justin.tv/cb/kinesis_processor/adapters"
	"code.justin.tv/cb/kinesis_processor/clients"
	"code.justin.tv/cb/kinesis_processor/clients/kinesis_firehose"
	"code.justin.tv/cb/kinesis_processor/models"
	"code.justin.tv/cb/kinesis_processor/utils"
	log "github.com/sirupsen/logrus"
)

const (
	ClipCreateParallelWorkersCount = 200
	ClipCreateFlushTreshold        = 50
)

//
// ClipCreate processor.
//
type ClipCreate interface {
	Before() error
	ProcessEvent(event kinesis_firehose.KinesisEvent) error
	After() error
}

type clipCreate struct {
	clients      *clients.Clients
	adapter      adapters.ClipsAdapter
	batchedClips map[string]*models.Clip

	wg      sync.WaitGroup
	records chan *models.Clip
}

// NewClipCreate create new processor.
func NewClipCreate(clients *clients.Clients) ClipCreate {
	return &clipCreate{
		clients:      clients,
		batchedClips: map[string]*models.Clip{},
		adapter:      adapters.NewClipsAdapter(clients.Conf.Environment, clients.Conf.AWSRegion),
	}
}

// Before - start workers
func (p *clipCreate) Before() error {
	// Setup worker pool
	p.records = make(chan *models.Clip, ClipCreateParallelWorkersCount)

	for w := 0; w < ClipCreateParallelWorkersCount; w++ {
		p.startWorker(w)
	}

	return nil
}

func (p *clipCreate) startWorker(number int) {
	p.wg.Add(1)

	go func(number int) {
		defer func() {
			p.wg.Done()
			if r := recover(); r != nil {
				log.Error("Failed worker: ", r)
			}

			// Chill for a minute.
			time.Sleep(1 * time.Minute)
			p.startWorker(number)
		}()

		for model := range p.records {
			err := p.adapter.UpdateCreateCount(context.TODO(), *model)
			if err != nil {
				log.Error(err)
				continue
			}
		}
	}(number)
}

// ProcessEvent
func (p *clipCreate) ProcessEvent(event kinesis_firehose.KinesisEvent) error {
	// clip id is required, if missing bail out
	if event.Fields["clip_id"] == "" {
		log.Warn("Received create clip event without clip_id set")
		return nil
	}

	clipID := event.Fields["clip_id"]

	channelID, err := strconv.ParseInt(event.Fields["channel_id"], 10, 64)
	if err != nil {
		return err
	}

	eventTime := event.Fields["time_utc"]
	if eventTime == "" {
		log.Error(errors.New("empty time_utc for create_clip event"))
		return nil
	}

	eventTimeTime, err := time.Parse(utils.DbTimeFormat, eventTime)
	if err != nil {
		return err
	}

	if _, ok := p.batchedClips[clipID]; !ok {
		p.batchedClips[clipID] = &models.Clip{
			ChannelID:      channelID,
			Time:           eventTimeTime,
			ClipID:         clipID,
			PlaysBreakdown: map[string]int64{},
		}
	}

	p.batchedClips[clipID].Creates++

	if len(p.batchedClips) > ClipCreateFlushTreshold {
		err := p.Flush()
		if err != nil {
			return err
		}
	}

	return nil
}

// After
func (p *clipCreate) After() error {
	close(p.records)
	p.wg.Wait()
	return nil
}

// Flush the buffer
func (p *clipCreate) Flush() error {
	for _, clip := range p.batchedClips {
		p.records <- clip
	}

	p.batchedClips = map[string]*models.Clip{}

	return nil
}
