package processors

import (
	"context"
	"strconv"
	"strings"
	"sync"
	"time"

	"code.justin.tv/cb/kinesis_processor/adapters"
	"code.justin.tv/cb/kinesis_processor/clients"
	"code.justin.tv/cb/kinesis_processor/clients/kinesis_firehose"
	"code.justin.tv/cb/kinesis_processor/models"
	log "github.com/sirupsen/logrus"
)

const (
	ClipVideoPlayParallelWorkersCount = 200
	clipVideoPlayFlushThreshold       = 500
)

//
// ClipVideoPlay processor.
//
type ClipVideoPlay interface {
	Before() error
	ProcessEvent(event kinesis_firehose.KinesisEvent) error
	After() error
}

type clipVideoPlay struct {
	clients      *clients.Clients
	adapter      adapters.ClipsAdapter
	batchedClips map[string]*models.Clip

	wg      sync.WaitGroup
	records chan *models.Clip
}

// NewClipVideoPlay create new processor.
func NewClipVideoPlay(clients *clients.Clients) ClipVideoPlay {
	return &clipVideoPlay{
		clients:      clients,
		batchedClips: map[string]*models.Clip{},
		adapter:      adapters.NewClipsAdapter(clients.Conf.Environment, clients.Conf.AWSRegion),
	}
}

// Before - start workers
func (p *clipVideoPlay) Before() error {

	// Setup worker pool
	p.records = make(chan *models.Clip, ClipVideoPlayParallelWorkersCount)

	for w := 0; w < ClipVideoPlayParallelWorkersCount; w++ {
		p.startWorker(w)
	}

	return nil
}

func (p *clipVideoPlay) startWorker(number int) {
	p.wg.Add(1)

	go func(number int) {
		defer func() {
			p.wg.Done()
			if r := recover(); r != nil {
				log.Error("Failed worker: ", r)
			}

			// Chill for a minute.
			time.Sleep(1 * time.Minute)
			p.startWorker(number)
		}()

		for model := range p.records {
			err := p.adapter.UpdateViewCount(context.TODO(), *model)
			if err != nil {
				log.Error(err)
				continue
			}
		}
	}(number)
}

// ProcessEvent
func (p *clipVideoPlay) ProcessEvent(event kinesis_firehose.KinesisEvent) error {
	// vod id is required, if missing bail out
	if event.Fields["vod_id"] == "" {
		log.Warn("Received video play event without vod_id set")
		return nil
	}

	clipID := event.Fields["vod_id"]

	channelID, err := strconv.ParseInt(event.Fields["channel_id"], 10, 64)
	if err != nil {
		return err
	}

	if _, ok := p.batchedClips[clipID]; !ok {
		p.batchedClips[clipID] = &models.Clip{
			ChannelID:      channelID,
			ClipID:         clipID,
			PlaysBreakdown: map[string]int64{},
		}
	}

	clip := p.batchedClips[clipID]
	clip.Plays++

	domain := getReferrer(event.Fields["referrer_url"], event.Fields["url"])
	clip.PlaysBreakdown[domain]++

	if len(p.batchedClips) > clipVideoPlayFlushThreshold {
		err := p.Flush()
		if err != nil {
			return err
		}
	}

	return nil
}

// After
func (p *clipVideoPlay) After() error {
	close(p.records)
	p.wg.Wait()
	return nil
}

// Flush the buffer
func (p *clipVideoPlay) Flush() error {
	for _, clip := range p.batchedClips {
		p.records <- clip
	}

	p.batchedClips = map[string]*models.Clip{}

	return nil
}

func getReferrer(referrerURL, url string) string {
	// Reddit and reddit with embedly
	if strings.Contains(referrerURL, ".reddit.") || strings.Contains(referrerURL, ".redd.it") || (strings.Contains(referrerURL, "embedly") && strings.Contains(url, "reddit")) {
		return adapters.RedditDomain
	}

	// Clips on channel page, games directory, my-clip, notifications, pulse recs, clips on page recommendations
	if strings.Contains(referrerURL, "www.twitch.tv") || strings.Contains(referrerURL, "my-clip") || strings.Contains(referrerURL, "clips.twitch.tv") {
		return adapters.TwitchDomain
	}

	// Facebook
	if strings.Contains(referrerURL, "facebook.") {
		return adapters.FacebookDomain
	}

	// Twitter
	if strings.Contains(referrerURL, "http://t.co") || strings.Contains(referrerURL, "https://t.co") || strings.Contains(referrerURL, "twitter") {
		return adapters.TwitterDomain
	}

	return adapters.OtherDomain
}
