package processors

import (
	"context"
	"strconv"
	"sync"
	"time"

	"code.justin.tv/cb/kinesis_processor/adapters"
	"code.justin.tv/cb/kinesis_processor/clients"
	"code.justin.tv/cb/kinesis_processor/clients/kinesis_firehose"
	"code.justin.tv/cb/kinesis_processor/models"
	"code.justin.tv/cb/kinesis_processor/utils"
	log "github.com/sirupsen/logrus"
)

const (
	CommercialParallelWorkersCount = 50
	CommercialBufferSize           = 25
)

type Commercial interface {
	Before() error
	ProcessEvent(event kinesis_firehose.KinesisEvent) error
	Flush() error
	After() error
}

type commercial struct {
	clients *clients.Clients
	adapter adapters.CommercialAdapter
	buffer  []models.Commercial
	wg      sync.WaitGroup
	records chan []models.Commercial
}

func NewCommercial(clients *clients.Clients) Commercial {
	return &commercial{
		clients: clients,
		adapter: adapters.NewCommercialAdapter(clients.Conf.Environment, clients.Conf.AWSRegion),
	}
}

// Before - start worksers
func (p *commercial) Before() error {
	p.buffer = []models.Commercial{}
	p.records = make(chan []models.Commercial, CommercialParallelWorkersCount)

	for w := 0; w < CommercialParallelWorkersCount; w++ {
		p.startWorker(w)
	}

	return nil
}

// startWorker - function called x amount of times for each go channel
func (p *commercial) startWorker(number int) {
	p.wg.Add(1)

	go func(number int) {
		defer func() {
			p.wg.Done()

			if r := recover(); r != nil {
				log.Error("Failed worker: ", r)
			}

			time.Sleep(1 * time.Minute)
			p.startWorker(number)
		}()

		var err error
		for models := range p.records {
			for _, model := range models {
				err = p.adapter.SaveCommercial(context.TODO(), model)
				if err != nil {
					if IsDuplicateError(err) {
						continue
					}
					log.Error(err)
					continue
				}
			}
		}
	}(number)
}

func (p *commercial) ProcessEvent(event kinesis_firehose.KinesisEvent) error {
	channelID, err := strconv.ParseInt(event.Fields["channel_id"], 10, 64)
	if err != nil {
		return nil
	}

	commercialLen, err := strconv.ParseInt(event.Fields["length"], 10, 64)
	if err != nil {
		return nil
	}

	var eventTime string
	if event.Fields["time_utc"] != "" {
		eventTime = event.Fields["time_utc"]
	} else {
		result, err := utils.ConvertPSTtoUTC(event.Fields["time"])
		if err != nil {
			return nil
		}
		eventTime = *result
	}

	eventTimeTime, err := time.Parse(utils.DbTimeFormat, eventTime)
	if err != nil {
		return err
	}

	dynamoDbTimestamp := &models.DynamoDBTimestamp{
		Converted: eventTimeTime,
	}

	obj := models.Commercial{
		ChannelID: channelID,
		Length:    commercialLen,
		Time:      *dynamoDbTimestamp,
	}

	p.buffer = append(p.buffer, obj)
	if len(p.buffer) >= CommercialBufferSize {
		err = p.Flush()
		if err != nil {
			return err
		}
	}

	return nil
}

// Flush - cleans out the buffer
func (p *commercial) Flush() error {
	tmp := make([]models.Commercial, len(p.buffer))
	copy(tmp, p.buffer)
	p.records <- tmp
	p.buffer = []models.Commercial{}

	return nil
}

func (p *commercial) After() error {
	err := p.Flush()
	if err != nil {
		return err
	}

	close(p.records)
	p.wg.Wait()
	return nil
}
