package processors

import (
	"context"
	"strconv"
	"sync"
	"time"

	"code.justin.tv/cb/kinesis_processor/adapters"
	"code.justin.tv/cb/kinesis_processor/clients"
	"code.justin.tv/cb/kinesis_processor/clients/kinesis_firehose"
	"code.justin.tv/cb/kinesis_processor/models"
	"code.justin.tv/cb/kinesis_processor/utils"
	log "github.com/sirupsen/logrus"
)

const (
	FollowAggregateUpdateParallelWorkersCount = 500
	FollowAggregateUpdateBufferSize           = 25
)

//
// FollowAggregateUpdate processor.
//
type FollowAggregateUpdate interface {
	Before() error
	ProcessEvent(event kinesis_firehose.KinesisEvent) error
	After() error
	Flush() error
}

type followAggregateUpdate struct {
	clients *clients.Clients
	adapter adapters.FollowAggregateUpdateAdapter
	buffer  []models.FollowAggregateUpdate

	wg          sync.WaitGroup
	records     chan []models.FollowAggregateUpdate
	deadCounter int
}

// NewFollowAggregateUpdate create new processor.
func NewFollowAggregateUpdate(clients *clients.Clients) FollowAggregateUpdate {
	return &followAggregateUpdate{
		clients: clients,
		adapter: adapters.NewFollowAggregateUpdateAdapter(clients.Conf.Environment, clients.Conf.AWSRegion),
	}
}

// Before - start workers
func (p *followAggregateUpdate) Before() error {
	p.buffer = []models.FollowAggregateUpdate{}
	// Setup worker pool
	p.records = make(chan []models.FollowAggregateUpdate, FollowAggregateUpdateParallelWorkersCount)

	for w := 0; w < FollowAggregateUpdateParallelWorkersCount; w++ {
		p.startWorker(w)
	}

	return nil
}

func (p *followAggregateUpdate) startWorker(number int) {
	p.wg.Add(1)

	go func(number int) {
		defer func() {
			p.wg.Done()
			if r := recover(); r != nil {
				log.Error("Failed worker: ", r)
			}
			// Chill for 1 minute
			time.Sleep(1 * time.Minute)
			p.startWorker(number)
		}()
		i := 0
		for models := range p.records {
			i++
			if i%1 == 0 {
				//fmt.Println("U[",number,"] ",i, len(p.records))
			}

			err := p.adapter.BatchSave(context.TODO(), models)
			if err != nil {
				if IsDuplicateError(err) {
					continue
				}
				log.Error(err)
				continue
			}
		}
	}(number)
}

// ProcessEvent
// Note that follow/unfollow events are frontend client values. The values are unreliabled
//	and it is not uncommon that the payload comes to us malformed or different depending on
//	client implementation -- if we cant parse the value rather than returning an error
//	we just return nil and skip the event altogether
func (p *followAggregateUpdate) ProcessEvent(event kinesis_firehose.KinesisEvent) error {
	channelID, err := strconv.ParseInt(event.Fields["channel_id"], 10, 64)
	if err != nil {
		return nil
	}

	total, err := strconv.ParseInt(event.Fields["follow_count"], 10, 64)
	if err != nil {
		return nil
	}

	timeStr, err := utils.ConvertPSTtoUTC(event.Fields["time"])
	if err != nil {
		return nil
	}

	// this on the other hand should not fail
	eventTimeTime, err := time.Parse(utils.DbTimeFormat, *timeStr)
	if err != nil {
		return err
	}

	model := models.FollowAggregateUpdate{
		ChannelID: channelID,
		Time:      eventTimeTime,
		Total:     total,
	}

	p.buffer = append(p.buffer, model)
	if len(p.buffer) >= FollowAggregateUpdateBufferSize {
		err = p.Flush()
		if err != nil {
			return err
		}
	}

	return nil
}

// After
func (p *followAggregateUpdate) After() error {
	err := p.Flush()
	if err != nil {
		return err
	}
	close(p.records)
	p.wg.Wait()
	return nil
}

// Flush
func (p *followAggregateUpdate) Flush() error {
	tmp := make([]models.FollowAggregateUpdate, len(p.buffer))
	copy(tmp, p.buffer)
	p.records <- tmp
	p.buffer = []models.FollowAggregateUpdate{}

	return nil
}
