package processors

import (
	"context"
	"strconv"
	"sync"
	"time"

	"code.justin.tv/cb/kinesis_processor/adapters"
	"code.justin.tv/cb/kinesis_processor/clients"
	"code.justin.tv/cb/kinesis_processor/clients/kinesis_firehose"
	"code.justin.tv/cb/kinesis_processor/models"
	"code.justin.tv/cb/kinesis_processor/utils"
	log "github.com/sirupsen/logrus"
)

const (
	IncrementalFollowParallelWorkersCount = 500
	IncrementalFollowBufferSize           = 25
)

//
// IncrementalFollow processor.
//
type IncrementalFollow interface {
	Before() error
	ProcessEvent(event kinesis_firehose.KinesisEvent) error
	After() error
	Flush() error
}

type incrementalFollow struct {
	clients *clients.Clients
	adapter adapters.IncrementalFollowAdapter
	buffer  []models.IncrementalFollow

	wg          sync.WaitGroup
	records     chan []models.IncrementalFollow
	deadCounter int
}

// NewIncrementalFollow create new processor.
func NewIncrementalFollow(clients *clients.Clients) IncrementalFollow {
	return &incrementalFollow{
		clients: clients,
		adapter: adapters.NewIncrementalFollowAdapter(clients.Conf.Environment, clients.Conf.AWSRegion),
	}
}

// Before - start workers
func (p *incrementalFollow) Before() error {
	p.buffer = []models.IncrementalFollow{}
	// Setup worker pool
	p.records = make(chan []models.IncrementalFollow, IncrementalFollowParallelWorkersCount)

	for w := 0; w < IncrementalFollowParallelWorkersCount; w++ {
		p.startWorker(w)
	}

	return nil
}

func (p *incrementalFollow) startWorker(number int) {
	p.wg.Add(1)

	go func(number int) {
		defer func() {
			p.wg.Done()
			if r := recover(); r != nil {
				log.Error("Failed worker: ", r)
			}
			// Chill for 1 minute
			time.Sleep(1 * time.Minute)
			p.startWorker(number)
		}()

		for models := range p.records {
			for _, model := range models {
				err := p.adapter.Save(context.TODO(), model)
				if err != nil {
					log.Error(err)
					continue
				}
			}
		}
	}(number)
}

// ProcessEvent
func (p *incrementalFollow) ProcessEvent(event kinesis_firehose.KinesisEvent) error {
	channelID, err := strconv.ParseInt(event.Fields["target_user_id"], 10, 64)
	if err != nil {
		return err
	}

	var eventTime string
	if event.Fields["time_utc"] != "" {
		eventTime = event.Fields["time_utc"]
	} else {
		result, err := utils.ConvertPSTtoUTC(event.Fields["time"])
		if err != nil {
			return err
		}
		eventTime = *result
	}

	eventTimeTime, err := time.Parse(utils.DbTimeFormat, eventTime)
	if err != nil {
		return err
	}

	model := models.IncrementalFollow{
		ChannelID: channelID,
		Time:      eventTimeTime,
	}

	p.buffer = append(p.buffer, model)
	if len(p.buffer) >= IncrementalFollowBufferSize {
		err = p.Flush()
		if err != nil {
			return err
		}
	}

	return nil
}

// After
func (p *incrementalFollow) After() error {
	err := p.Flush()
	if err != nil {
		return err
	}
	close(p.records)
	p.wg.Wait()
	return nil
}

// Flush
func (p *incrementalFollow) Flush() error {
	tmp := make([]models.IncrementalFollow, len(p.buffer))
	copy(tmp, p.buffer)
	p.records <- tmp
	p.buffer = []models.IncrementalFollow{}

	return nil
}
