package processors

import (
	"context"
	"errors"
	"strconv"
	"sync"
	"time"

	"code.justin.tv/cb/kinesis_processor/adapters"
	"code.justin.tv/cb/kinesis_processor/clients"
	"code.justin.tv/cb/kinesis_processor/clients/kinesis_firehose"
	"code.justin.tv/cb/kinesis_processor/models"
	"code.justin.tv/cb/kinesis_processor/utils"
	log "github.com/sirupsen/logrus"
)

const (
	MinuteBroadcastParallelWorkersCount = 500
	MinuteBroadcastBufferSize           = 25
)

// ErrInvalidMinuteBroadcastEvent is the error returned
// when a "minute broadcast" event is malformed due to missing
// field(s) or invalid field type(s).
var ErrInvalidMinuteBroadcastEvent = errors.New("processors: malformed minute broadcast event")

//
// MinuteBroadcast processor.
//
type MinuteBroadcast interface {
	Before() error
	ProcessEvent(event kinesis_firehose.KinesisEvent) error
	After() error
	Flush() error
}

type minuteBroadcast struct {
	clients          *clients.Clients
	adapter          adapters.MinuteBroadcastAdapter
	buffer           []models.MinuteBroadcast

	wg          sync.WaitGroup
	records     chan []models.MinuteBroadcast
	deadCounter int
}

// NewMinuteBroadcast create new processor.
func NewMinuteBroadcast(clients *clients.Clients) MinuteBroadcast {
	return &minuteBroadcast{
		clients:          clients,
		adapter:          adapters.NewMinuteBroadcastAdapter(clients.Conf.Environment, clients.Conf.AWSRegion),
	}
}

// Before - start workers
func (p *minuteBroadcast) Before() error {
	p.buffer = []models.MinuteBroadcast{}
	// Setup worker pool
	p.records = make(chan []models.MinuteBroadcast, MinuteBroadcastParallelWorkersCount)

	for w := 0; w < MinuteBroadcastParallelWorkersCount; w++ {
		p.startWorker(w)
	}

	return nil
}

func (p *minuteBroadcast) startWorker(number int) {
	p.wg.Add(1)

	go func(number int) {
		defer func() {
			p.wg.Done()
			if r := recover(); r != nil {
				log.Error("Failed worker: ", r)
			}
			// Chill for 1 minute
			time.Sleep(1 * time.Minute)
			p.startWorker(number)
		}()
		i := 0
		for models := range p.records {
			i++
			if i%1 == 0 {
				//fmt.Println("U[",number,"] ",i, len(p.records))
			}

			err := p.adapter.BatchSave(context.TODO(), models)
			if err != nil {
				if IsDuplicateError(err) {
					continue
				}
				log.Error(err)
				continue
			}
		}
	}(number)
}

// ProcessEvent
func (p *minuteBroadcast) ProcessEvent(event kinesis_firehose.KinesisEvent) error {
	// Preparing fields for MinuteBroadcast model.
	channelID, err := strconv.ParseInt(event.Fields["channel_id"], 10, 64)
	if err != nil {
		return ErrInvalidMinuteBroadcastEvent
	}

	broadcastID, err := strconv.ParseInt(event.Fields["broadcast_id"], 10, 64)
	if err != nil {
		return ErrInvalidMinuteBroadcastEvent
	}

	var eventTime string
	if event.Fields["time_utc"] != "" {
		eventTime = event.Fields["time_utc"]
	} else {
		result, err := utils.ConvertPSTtoUTC(event.Fields["time"])
		if err != nil {
			return err
		}
		eventTime = *result
	}

	eventTimeTime, err := time.Parse(utils.DbTimeFormat, eventTime)
	if err != nil {
		return err
	}

	model := models.MinuteBroadcast{
		ChannelID:           channelID,
		Time:                eventTimeTime,
		Game:                event.Fields["game"],
		BroadcastID:         broadcastID,
		BroadcasterSoftware: event.Fields["broadcaster_software"],
	}

	p.buffer = append(p.buffer, model)
	if len(p.buffer) >= MinuteBroadcastBufferSize {
		err = p.Flush()
		if err != nil {
			return err
		}
	}

	return nil
}

// After
func (p *minuteBroadcast) After() error {
	err := p.Flush()
	close(p.records)
	p.wg.Wait()
	return err
}

// Flush
func (p *minuteBroadcast) Flush() error {
	tmp := make([]models.MinuteBroadcast, len(p.buffer))
	copy(tmp, p.buffer)
	p.records <- tmp
	p.buffer = []models.MinuteBroadcast{}

	return nil
}
