package processors

import (
	"context"
	"strconv"
	"sync"
	"time"

	"code.justin.tv/cb/kinesis_processor/adapters"
	"code.justin.tv/cb/kinesis_processor/clients"
	"code.justin.tv/cb/kinesis_processor/clients/kinesis_firehose"
	"code.justin.tv/cb/kinesis_processor/models"
	"code.justin.tv/cb/kinesis_processor/utils"
	log "github.com/sirupsen/logrus"
)

const (
	RaidParallelWorkersCount = 50
	RaidBufferSize           = 25
)

type Raid interface {
	Before() error
	ProcessEvent(event kinesis_firehose.KinesisEvent) error
	Flush() error
	After() error
}

type raid struct {
	clients *clients.Clients
	adapter adapters.RaidsAdapter
	buffer  []models.Raid
	wg      sync.WaitGroup
	records chan []models.Raid
}

func NewRaid(clients *clients.Clients) Raid {
	return &raid{
		clients: clients,
		adapter: adapters.NewRaidsAdapter(clients.Conf.Environment, clients.Conf.AWSRegion),
	}
}

// Before - start worksers
func (p *raid) Before() error {
	p.buffer = []models.Raid{}
	p.records = make(chan []models.Raid, RaidParallelWorkersCount)

	for w := 0; w < RaidParallelWorkersCount; w++ {
		p.startWorker(w)
	}

	return nil
}

// startWorker - function called x amount of times for each go channel
func (p *raid) startWorker(number int) {
	p.wg.Add(1)

	go func(number int) {
		defer func() {
			p.wg.Done()

			if r := recover(); r != nil {
				log.Error("Failed worker: ", r)
			}

			time.Sleep(1 * time.Minute)
			p.startWorker(number)
		}()

		var err error
		for models := range p.records {
			for _, model := range models {
				err = p.adapter.SaveRaid(context.TODO(), model)
				if err != nil {
					if IsDuplicateError(err) {
						continue
					}
					log.Error(err)
					continue
				}
			}
		}
	}(number)
}

func (p *raid) ProcessEvent(event kinesis_firehose.KinesisEvent) error {
	viewerCount, err := strconv.ParseInt(event.Fields["viewer_count"], 10, 64)
	if err != nil {
		return err
	}

	var eventTime string
	if event.Fields["time_utc"] != "" {
		eventTime = event.Fields["time_utc"]
	} else {
		result, err := utils.ConvertPSTtoUTC(event.Fields["time"])
		if err != nil {
			return nil
		}
		eventTime = *result
	}

	eventTimeTime, err := time.Parse(utils.DbTimeFormat, eventTime)
	if err != nil {
		return err
	}

	dynamoDbTimestamp := &models.DynamoDBTimestamp{
		Converted: eventTimeTime,
	}

	obj := models.Raid{
		RaidID:      event.Fields["raid_id"],
		TargetID:    event.Fields["raid_target_id"],
		ViewerCount: viewerCount,
		Time:        *dynamoDbTimestamp,
	}

	p.buffer = append(p.buffer, obj)
	if len(p.buffer) >= RaidBufferSize {
		err = p.Flush()
		if err != nil {
			return err
		}
	}

	return nil
}

// Flush - cleans out the buffer
func (p *raid) Flush() error {
	tmp := make([]models.Raid, len(p.buffer))
	copy(tmp, p.buffer)
	p.records <- tmp
	p.buffer = []models.Raid{}

	return nil
}

func (p *raid) After() error {
	err := p.Flush()
	if err != nil {
		return err
	}

	close(p.records)
	p.wg.Wait()
	return nil
}
