package processors

import (
	"context"
	"net/url"
	"strconv"
	"sync"
	"time"

	"code.justin.tv/cb/kinesis_processor/adapters"
	"code.justin.tv/cb/kinesis_processor/clients"
	"code.justin.tv/cb/kinesis_processor/clients/kinesis_firehose"
	"code.justin.tv/cb/kinesis_processor/models"
	"code.justin.tv/cb/kinesis_processor/utils"
	log "github.com/sirupsen/logrus"
)

const (
	VideoPlayParallelWorkersCount = 200
	VideoPlayBufferSize           = 25
)

//
// ClipCreate processor.
//
type VideoPlay interface {
	Before() error
	Flush() error
	ProcessEvent(event kinesis_firehose.KinesisEvent) error
	After() error
}

type videoPlay struct {
	clients *clients.Clients
	adapter adapters.VideoPlayAdapter
	buffer  []models.VideoPlay

	wg          sync.WaitGroup
	records     chan []models.VideoPlay
	deadCounter int
}

// NewVideoPlay create new processor.
func NewVideoPlay(clients *clients.Clients) VideoPlay {
	return &videoPlay{
		clients: clients,
		adapter: adapters.NewVideoPlayAdapter(clients.Conf.Environment, clients.Conf.AWSRegion),
	}
}

// Before - start workers
func (p *videoPlay) Before() error {
	p.buffer = []models.VideoPlay{}
	// Setup worker pool
	p.records = make(chan []models.VideoPlay, VideoPlayParallelWorkersCount)

	for w := 0; w < VideoPlayParallelWorkersCount; w++ {
		p.startWorker(w)
	}

	return nil
}

func (p *videoPlay) startWorker(number int) {
	p.wg.Add(1)

	go func(number int) {
		defer func() {
			p.wg.Done()
			if r := recover(); r != nil {
				log.Error("Failed worker: ", r)
			}
			// Chill for 1 minute
			time.Sleep(1 * time.Minute)
			p.startWorker(number)
		}()

		for models := range p.records {
			for _, model := range models {
				if model.Live {
					err := p.adapter.SaveGeoData(context.TODO(), model)
					if err != nil {
						log.Error("Failed caching geo data: ", err)
					}

					err = p.adapter.SavePlatformData(context.TODO(), model)
					if err != nil {
						log.Error("Failed caching platform data: ", err)
					}

					if model.Platform == "web" {
						err = p.adapter.SaveReferralData(context.TODO(), model)
						if err != nil {
							log.Error("Failed caching referral data: ", err)
						}
					}
					// TODO: Cam - Comment out unique views table is ready for production
					// err = p.adapter.SaveDevicesData(context.TODO(), model)
					// if err != nil {
					// 	log.Error("Failed caching unique-plays data: ", err)
					// }
				}
			}
		}
	}(number)
}

// ProcessEvent
func (p *videoPlay) ProcessEvent(event kinesis_firehose.KinesisEvent) error {
	// this is one of the our keys so should this parse fails we skip the event
	channelID, err := strconv.ParseInt(event.Fields["channel_id"], 10, 64)
	if err != nil {
		return nil
	}

	var eventTime string
	if event.Fields["time_utc"] != "" {
		eventTime = event.Fields["time_utc"]
	} else {
		result, err := utils.ConvertPSTtoUTC(event.Fields["time"])
		if err != nil {
			return nil
		}
		eventTime = *result
	}

	// this is one of the our keys so should this parse fails we skip the event
	eventTimeTime, err := time.Parse(utils.DbTimeFormat, eventTime)
	if err != nil {
		return nil
	}

	hostChannelID, err := strconv.ParseInt(event.Fields["host_channel_id"], 10, 64)
	if err != nil {
		// we interpret a garbage host channel value as 0 (an invalid id)
		hostChannelID = 0
	}

	live, err := strconv.ParseBool(event.Fields["live"])
	if err != nil {
		// we interpret null or any other garbage values in spade as true
		live = true
	}

	referrer, err := url.Parse(event.Fields["referrer"])
	if err != nil {
		// we interpret an invalid referrer as nil (no referrer)
	}

	externalReferrer, err := url.Parse(event.Fields["referrer_url"])
	if err != nil {
		// we interpret an invalid referrer as nil (no referrer)
	}

	obj := models.VideoPlay{
		ChannelID:     channelID,
		Time:          eventTimeTime,
		HostChannelID: hostChannelID,
		Referrer:      referrer,
		ReferrerHost:  event.Fields["referrer_host"],
		ReferrerUrl:   externalReferrer,
		Live:          live,
		Content:       event.Fields["content"],
		ContentMode:   event.Fields["content_mode"],
		Player:        event.Fields["player"],
		Country:       event.Fields["country"],
		City:          event.Fields["city"],
		Medium:        event.Fields["medium"],
		Platform:      event.Fields["platform"],
		DeviceID:      event.Fields["device_id"],
	}

	p.buffer = append(p.buffer, obj)
	if len(p.buffer) >= VideoPlayBufferSize {
		err = p.Flush()
		if err != nil {
			return err
		}
	}

	return nil
}

// After
func (p *videoPlay) After() error {
	err := p.Flush()
	if err != nil {
		return err
	}
	close(p.records)
	p.wg.Wait()
	return nil
}

// Flush the buffer
func (p *videoPlay) Flush() error {
	tmp := make([]models.VideoPlay, len(p.buffer))
	copy(tmp, p.buffer)
	p.records <- tmp
	p.buffer = []models.VideoPlay{}

	return nil
}
