package database

import (
	"time"

	"code.justin.tv/cb/kinesis_processor/models"
	"github.com/jmoiron/sqlx"
	log "github.com/sirupsen/logrus"
)

// Database is the interface that exposes functions for accessing the database.
type Database interface {
	// CheckDBConnection calls on sql.Ping() to verify connection.
	//
	// If there is no connection, sql.Ping() attempts to establish a connection.
	// An error is returned if the database connection attempt fails.
	CheckDBConnection() error
	// InsertChannelUpdate creates a record in 'channel_updates' table
	// given models.ChannelUpdate and return inserted ID
	// or error if anything goes wrong.
	InsertChannelUpdate(model models.ChannelUpdate) (int, error)
	// BulkInsertChannelUpdate creates multiple records in 'channel_updates' table
	// given []models.ChannelUpdate and return error if anything goes wrong.
	BulkInsertChannelUpdate(models []models.ChannelUpdate) error

	// InsertChannelConcurrent creates a record in 'channel_concurrents' table
	// given models.ChannelConcurrent and return inserted ID
	// or error if anything goes wrong.
	InsertChannelConcurrent(model models.ChannelConcurrent) (int64, error)
	// BulkInsertChannelConcurrent creates multiple records in 'channel_concurrents' table
	// given []models.ChannelConcurrent and return error if anything goes wrong.
	BulkInsertChannelConcurrent(models []models.ChannelConcurrent) error

	// SelectExtandableChannelSession get ChannelSession that time is within 5min
	// window, given channelID and time and return models.ChannelSession if found
	// or error if anything goes wrong.
	SelectExtendableChannelSession(channelID int64, time time.Time) (*models.ChannelSession, error)
	// ExtendChannelSession extends end_time if new end_time is within 5min
	// given channelID and endTime and return affected rows count
	// or error if anything goes wrong.
	ExtendChannelSession(channelID int64, endTime time.Time) (int64, error)
	// InsertChannelSession creates a record in 'channel_sessions' table
	// given models.ChannelSession and return inserted ID
	// or error if anything goes wrong.
	InsertChannelSession(model models.ChannelSession) (int64, error)
	// Update update models.ChannelSession in channel_sessions
	// using ID and returns error if anything goes wrong.
	UpdateChannelSession(model models.ChannelSession) (int64, error)

	// InsertChannelSessionBroadcast creates a record in 'channel_session_broadcasts' table
	// given sessionID and array of broadcastIDs []int64
	// or error if anything goes wrong.
	InsertChannelSessionBroadcasts(sessionID int64, broadcastIDs []int64) error
}

// database is a wrapped database handle for the connection pool to AWS RDS.
type database struct {
	*sqlx.DB
}

// NewDBConnection establishes a TCP connection to the AWS RDS cluster,
// and instantiates a database struct wrapping the database connection pool.
// It returns the exported Database interface.
func NewDBConnection(driver, source string) (Database, error) {
	db, err := sqlx.Open(driver, source)
	if err != nil {
		log.WithError(err).Error("Failed to open connection to database")
		return nil, err
	}

	return &database{db}, nil
}

// CheckDBConnection calls on sql.Ping() to verify a database connection.
func (db *database) CheckDBConnection() error {
	err := db.Ping()
	if err != nil {
		log.WithError(err).Error("Failed to verify connection to database")

		return err
	}

	return nil
}
