package database

import (
	"code.justin.tv/cb/kinesis_processor/models"
	log "github.com/sirupsen/logrus"
)

// InsertChannelConcurrent creates a record in 'channel_concurrents' table
// given models.ChannelConcurrent and return inserted ID
// or error if anything goes wrong.
func (db *database) InsertChannelConcurrent(model models.ChannelConcurrent) (int64, error) {
	statement := `
			INSERT INTO channel_concurrents
			(channel_id, time, total, content_mode)
			VALUES(:channel_id, :time, :total, :content_mode);
		`

	res, err := db.NamedExec(statement, model)
	if err != nil {
		msg := "Failed to execute InsertChannelConcurrent statement"
		log.WithError(err).Warn(msg)
		return 0, err
	}

	id, err := res.LastInsertId()
	if err != nil {
		msg := "Failed to return last insert ID for InsertChannelConcurrent statement"
		log.WithError(err).Warn(msg)
		return 0, err
	}

	return id, nil
}

// BulkInsertChannelConcurrent creates multiple records in 'channel_concurrents' table
// given []models.ChannelConcurrent and return error if anything goes wrong.
func (db *database) BulkInsertChannelConcurrent(models []models.ChannelConcurrent) error {
	tx := db.MustBegin()

	for _, model := range models {
		statement := `
			INSERT INTO channel_concurrents
			(channel_id, time, total, content_mode)
			VALUES(:channel_id, :time, :total, :content_mode);
		`

		_, err := tx.NamedExec(statement, &model)
		if err != nil {
			msg := "Failed to execute BulkInsertChannelConcurrents statement."
			log.WithError(err).Warn(msg)
			return err
		}
	}
	err := tx.Commit()
	if err != nil {
		msg := "Failed to execute BulkInsertChannelConcurrents statement."
		log.WithError(err).Warn(msg)
		return err
	}

	return nil
}
