package database

import (
	"code.justin.tv/cb/kinesis_processor/models"
	log "github.com/sirupsen/logrus"
)

// InsertChannelUpdate creates a record in 'channel_updates' table
// given models.ChannelUpdate and return inserted ID
// or error if anything goes wrong.
func (db *database) InsertChannelUpdate(model models.ChannelUpdate) (int, error) {
	statement := `
		INSERT INTO channel_updates
		(channel_id, time, title, old_title, game, old_game, status, old_status, broadcaster_language, old_broadcaster_language)
		VALUES(:channel_id, :time, :title, :old_title, :game, :old_game, :status, :old_status, :broadcaster_language, :old_broadcaster_language);
	`

	res, err := db.NamedExec(statement, model)
	if err != nil {
		msg := "Failed to execute InsertChannelUpdate statement"
		log.WithError(err).Warn(msg)
		return 0, err
	}

	id, err := res.LastInsertId()
	if err != nil {
		msg := "Failed to return last insert ID for InsertChannelUpdate statement"
		log.WithError(err).Warn(msg)
		return 0, err
	}

	return int(id), nil
}

// BulkInsertChannelUpdate creates multiple records in 'channel_updates' table
// given []models.ChannelUpdate and return error if anything goes wrong.
func (db *database) BulkInsertChannelUpdate(models []models.ChannelUpdate) error {
	tx := db.MustBegin()

	for _, model := range models {
		statement := `
			INSERT INTO channel_updates
			(channel_id, time, title, old_title, game, old_game, status, old_status, broadcaster_language, old_broadcaster_language)
			VALUES(:channel_id, :time, :title, :old_title, :game, :old_game, :status, :old_status, :broadcaster_language, :old_broadcaster_language);
		`

		_, err := tx.NamedExec(statement, &model)
		if err != nil {
			msg := "Failed to execute BulkInsertChannelUpdate statement"
			log.WithError(err).Warn(msg)
			return err
		}
	}
	err := tx.Commit()
	if err != nil {
		msg := "Failed to execute BulkInsertChannelUpdate statement"
		log.WithError(err).Warn(msg)
		return err
	}

	return nil
}
