package dynamo_adapter

import (
	"fmt"
	"strconv"

	"code.justin.tv/cb/kinesis_processor/config"
	"code.justin.tv/cb/kinesis_processor/models"
	"code.justin.tv/cb/kinesis_processor/utils"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"
)

const (
	TableChannelSessions    = "CbChannelSessions"
	TableChannelConcurrents = "CbChannelConcurrents"
	TableChannelUpdates     = "CbChannelUpdates"
)

// DynamoAdapter is the interface that exposes functions for accessing the database.
type DynamoAdapter interface {
	AddChannelUpdate(obj models.ChannelUpdate) error
	AddChannelConcurrents(obj models.ChannelConcurrent) error
	AddChannelSession(obj models.ChannelSession) error
	FetchMinuteBrodcast(exclusiveStartKey map[string]*dynamodb.AttributeValue) ([]models.MinuteBroadcast, map[string]*dynamodb.AttributeValue, error)
	FetchMinuteBrodcastByChannelID(channelID int64, exclusiveStartKey map[string]*dynamodb.AttributeValue) ([]models.MinuteBroadcast, map[string]*dynamodb.AttributeValue, error)
	GetLastMinutesBroadcast(minutes int) (*[]models.MinuteBroadcast, error)
}

// dynamoAdapter is a wrapped database handle for the connection pool to AWS RDS.
type dynamoAdapter struct {
	client dynamodbiface.DynamoDBAPI
}

// NewDynamoAdapter new client to DynamoDB.
func NewDynamoAdapter(conf *config.Config) (DynamoAdapter, error) {
	credentials := conf.NewAWSCredentials()

	awsConfig := &aws.Config{
		Region:           aws.String(conf.AWSRegion),
		S3ForcePathStyle: aws.Bool(true),
		Credentials:      credentials,
	}

	return &dynamoAdapter{
		client: dynamodb.New(session.New(awsConfig)),
	}, nil
}

// AddChannelUpdate just add new record in CbChannelUpdates
func (d *dynamoAdapter) AddChannelUpdate(obj models.ChannelUpdate) error {
	itemAttributes := map[string]*dynamodb.AttributeValue{
		"ChannelID": &dynamodb.AttributeValue{N: aws.String(strconv.FormatInt(obj.ChannelID, 10))},
		"Time":      &dynamodb.AttributeValue{S: aws.String(obj.Time.Format(utils.DbTimeFormat))},
	}

	if obj.Game != "" {
		itemAttributes["Game"] = &dynamodb.AttributeValue{S: aws.String(obj.Game)}
	}
	if obj.OldGame != "" {
		itemAttributes["OldGame"] = &dynamodb.AttributeValue{S: aws.String(obj.OldGame)}
	}
	if obj.GameID != "" {
		itemAttributes["GameID"] = &dynamodb.AttributeValue{S: aws.String(obj.GameID)}
	}
	if obj.OldGameID != "" {
		itemAttributes["OldGameID"] = &dynamodb.AttributeValue{S: aws.String(obj.OldGameID)}
	}
	if obj.Status != "" {
		itemAttributes["Status"] = &dynamodb.AttributeValue{S: aws.String(obj.Status)}
	}
	if obj.OldStatus != "" {
		itemAttributes["OldStatus"] = &dynamodb.AttributeValue{S: aws.String(obj.OldStatus)}
	}
	if obj.BroadcasterLanguage != "" {
		itemAttributes["BroadcastLanguage"] = &dynamodb.AttributeValue{S: aws.String(obj.BroadcasterLanguage)}
	}
	if obj.OldBroadcasterLanguage != "" {
		itemAttributes["OldBroadcastLanguage"] = &dynamodb.AttributeValue{S: aws.String(obj.OldBroadcasterLanguage)}
	}

	_, err := d.client.PutItem(&dynamodb.PutItemInput{
		Item:      itemAttributes,
		TableName: aws.String(TableChannelUpdates),
	})
	if err != nil {
		fmt.Println(err)
		fmt.Println(obj)
	}

	return err
}

// AddChannelConcurrents add to CbChannelConcurrents
func (d *dynamoAdapter) AddChannelConcurrents(obj models.ChannelConcurrent) error {
	_, err := d.client.PutItem(&dynamodb.PutItemInput{
		Item: map[string]*dynamodb.AttributeValue{
			"ChannelID":   &dynamodb.AttributeValue{N: aws.String(strconv.FormatInt(obj.ChannelID, 10))},
			"Time":        &dynamodb.AttributeValue{S: aws.String(obj.Time.Format(utils.DbTimeFormat))},
			"Total":       &dynamodb.AttributeValue{N: aws.String(strconv.FormatInt(obj.Total, 10))},
			"ContentMode": &dynamodb.AttributeValue{S: aws.String(obj.ContentMode)},
		},
		TableName: aws.String(TableChannelConcurrents),
	})

	return err
}
