package dynamo_adapter

import (
	"code.justin.tv/cb/kinesis_processor/adapters"
	"code.justin.tv/cb/kinesis_processor/models"
	"code.justin.tv/cb/kinesis_processor/utils"
	"fmt"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"strconv"
	"time"
)

// FetchMinuteBrodcast fetch all minutes broadcast.
func (d *dynamoAdapter) FetchMinuteBrodcast(exclusiveStartKey map[string]*dynamodb.AttributeValue) ([]models.MinuteBroadcast, map[string]*dynamodb.AttributeValue, error) {

	result := []models.MinuteBroadcast{}
	output, err := d.client.Scan(&dynamodb.ScanInput{
		//FilterExpression: aws.String("ChannelID = :channelID"),
		//ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
		//	":channelID": {
		//		N: aws.String("151859374"),
		//	},
		//},

		TableName:         aws.String(adapters.TableMinuteBroadcasts),
		ExclusiveStartKey: exclusiveStartKey,
		Limit:             aws.Int64(100),
	})

	if err != nil {
		return nil, nil, err
	}

	for _, value := range output.Items {
		model, err := buildMinuteBroadcastModel(value)
		if err != nil {
			return nil, nil, err
		}

		result = append(result, *model)
	}

	return result, output.LastEvaluatedKey, nil
}

// FetchMinuteBroadcastByChannelID
func (d *dynamoAdapter) FetchMinuteBrodcastByChannelID(channelID int64, exclusiveStartKey map[string]*dynamodb.AttributeValue) ([]models.MinuteBroadcast, map[string]*dynamodb.AttributeValue, error) {
	result := []models.MinuteBroadcast{}
	output, err := d.client.Query(&dynamodb.QueryInput{
		KeyConditionExpression: aws.String("ChannelID = :channelID"),
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":channelID": {
				N: aws.String(strconv.FormatInt(channelID, 10)),
			},
		},

		ScanIndexForward:  aws.Bool(false),
		TableName:         aws.String(adapters.TableMinuteBroadcasts),
		ExclusiveStartKey: exclusiveStartKey,
		Limit:             aws.Int64(100),
	})

	if err != nil {
		return nil, nil, err
	}

	for _, value := range output.Items {
		model, err := buildMinuteBroadcastModel(value)
		if err != nil {
			return nil, nil, err
		}

		result = append(result, *model)
	}

	return result, output.LastEvaluatedKey, nil
}

// GetLastChannelSession
func (d *dynamoAdapter) GetLastMinutesBroadcast(minutes int) (*[]models.MinuteBroadcast, error) {
	// this is for pagination
	var exclusiveStartKey map[string]*dynamodb.AttributeValue

	result := []models.MinuteBroadcast{}
	for {
		output, err := d.client.Query(&dynamodb.QueryInput{
			IndexName:              aws.String("TimeIndex"),
			KeyConditionExpression: aws.String("#T >= :time"),
			ExpressionAttributeNames: map[string]*string{
				"#T": aws.String("Time"),
			},
			ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
				":time": {
					S: aws.String(time.Now().Add(time.Duration(-minutes) * time.Minute).Format(utils.DbTimeFormat)),
				},
			},

			TableName:         aws.String(adapters.TableMinuteBroadcasts),
			ExclusiveStartKey: exclusiveStartKey,
			Limit:             aws.Int64(1000),
		})

		if err != nil {
			return nil, err
		}

		for _, value := range output.Items {
			model, err := buildMinuteBroadcastModel(value)
			if err != nil {
				return nil, err
			}

			result = append(result, *model)
		}
		fmt.Println("Loaded models", len(output.Items), "/", len(result))

		if output.LastEvaluatedKey == nil {
			break
		}

		exclusiveStartKey = output.LastEvaluatedKey
	}
	return &result, nil
}

func buildMinuteBroadcastModel(value map[string]*dynamodb.AttributeValue) (*models.MinuteBroadcast, error) {
	channelID, err := strconv.ParseInt(*value["ChannelID"].N, 10, 64)
	if err != nil {
		return nil, err
	}

	timetime, err := time.Parse(utils.DbTimeFormat, *value["Time"].S)
	if err != nil {
		return nil, err
	}

	broadcastID, err := strconv.ParseInt(*value["BroadcastID"].N, 10, 64)
	if err != nil {
		return nil, err
	}

	var game string
	if value["Game"] != nil {
		game = *value["Game"].S
	}

	var software string
	if value["BroadcasterSoftware"] != nil {
		software = *value["BroadcasterSoftware"].S
	}

	return &models.MinuteBroadcast{
		ChannelID:           channelID,
		Time:                timetime,
		Game:                game,
		BroadcastID:         broadcastID,
		BroadcasterSoftware: software,
	}, nil
}
