package kinesis_firehose

import (
	"bufio"
	"bytes"
	"compress/flate"
	"encoding/json"
	"errors"
	"fmt"
	"io"

	"code.justin.tv/cb/kinesis_processor/config"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/aws/aws-sdk-go/service/s3/s3iface"
	log "github.com/sirupsen/logrus"
)

//
// KinesisFirehose is used as client for Kinesis Firehouse. D'oh
//
type KinesisFirehose interface {
	Get() (*KinesisContainer, error)
	DeleteObjectByKey(key string) error
	FailObjectByKey(key string) error
}

type kinesisFirehose struct {
	client     s3iface.S3API
	bucketName string
	prefix     string
}

// NewKinesisFirehose create S3Storage instance with S3 client in it
func NewKinesisFirehose(conf *config.Config, bucketName string, prefix string) KinesisFirehose {
	credentials := conf.NewAWSCredentials()

	awsConfig := &aws.Config{
		Region:           aws.String(conf.AWSRegion),
		S3ForcePathStyle: aws.Bool(true),
		Credentials:      credentials,
	}

	return &kinesisFirehose{
		client:     s3.New(session.New(awsConfig)),
		bucketName: bucketName,
		prefix:     prefix,
	}
}

func (k *kinesisFirehose) Get() (*KinesisContainer, error) {
	key, reader, err := k.FetchFirstObject()

	// Nothing
	if key == nil && reader == nil && err == nil {
		return &KinesisContainer{
			Key:    "",
			Events: []KinesisEvent{},
		}, nil
	}

	// Error processing
	if err != nil {
		return nil, err
	}

	result := KinesisContainer{
		Key: *key,
	}

	// Read blob and scan by lines
	scanner := bufio.NewScanner(*reader)
	buf := make([]byte, 0, 64*1024)
	scanner.Buffer(buf, 1024*1024)
	for scanner.Scan() {
		// Read Kinesis Record
		var kinesisRecord KinesisRecord
		err := json.Unmarshal(scanner.Bytes(), &kinesisRecord)
		if err != nil {
			return nil, errors.New(fmt.Sprintf("json Unmarshal error: %v", err))
		}

		// Validate version
		if kinesisRecord.Version != 1 {
			return nil, errors.New(fmt.Sprintf("kinesis event version was not 1, instead: %v", kinesisRecord.Version))
		}

		// Validate first byte of data == 1
		if kinesisRecord.Data[0] != 1 {
			return nil, errors.New(fmt.Sprintf("data[0] was not 1, instead: %v", kinesisRecord.Data[0]))
		}

		// Read KinesisEvents from KinesisRecord.Data
		deflator := flate.NewReader(bytes.NewBuffer(kinesisRecord.Data[1:]))
		defer func() {
			err = deflator.Close()
			if err != nil {
				log.Error(errors.New(fmt.Sprintf("Error closing deflator: %v", err)))
			}
		}()

		var decompressed bytes.Buffer
		_, err = io.Copy(&decompressed, deflator)
		if err != nil {
			return nil, errors.New(fmt.Sprintf("Error decompressing: %v", err))
		}

		var kinesisEvents []KinesisEvent
		err = json.Unmarshal(decompressed.Bytes(), &kinesisEvents)
		if err != nil {
			return nil, errors.New(fmt.Sprintf("json Unmarshal error: %v", err))
		}
		result.Events = append(result.Events, kinesisEvents...)

	}
	if err := scanner.Err(); err != nil {
		return nil, err
	}

	return &result, nil
}

func (k *kinesisFirehose) FetchFirstObject() (*string, *io.ReadCloser, error) {
	// get first object out there
	listInput := &s3.ListObjectsV2Input{
		Bucket:  aws.String(k.bucketName),
		Prefix:  aws.String(k.prefix),
		MaxKeys: aws.Int64(1),
	}
	resp, err := k.client.ListObjectsV2(listInput)
	if err != nil {
		return nil, nil, err
	}

	// if no keys returned
	if *resp.KeyCount == 0 {
		return nil, nil, nil
	}

	objectOutput, err := k.client.GetObject(&s3.GetObjectInput{
		Bucket: aws.String(k.bucketName),
		Key:    resp.Contents[0].Key,
	})
	if err != nil {
		return nil, nil, err
	}

	return resp.Contents[0].Key, &objectOutput.Body, nil

}

func (k *kinesisFirehose) FailObjectByKey(key string) error {
	_, err := k.client.CopyObject(&s3.CopyObjectInput{
		Bucket:     aws.String(k.bucketName),             // Bucket To
		CopySource: aws.String(k.bucketName + "/" + key), // Source bucket+/+key = cb-kinesis-s3/incomming/2016.../cb-analytics-1-2016-11-30-22-00-26-45c31b11-a974-41ad-b2c3-2b1dc8864465
		Key:        aws.String("failed_" + key),          // failed_incomming/2016.../cb-analytics-1-2016-11-30-22-00-26-45c31b11-a974-41ad-b2c3-2b1dc8864465
	})

	if err != nil {
		return err
	}

	return k.DeleteObjectByKey(key)
}

func (k *kinesisFirehose) DeleteObjectByKey(key string) error {
	deleteInput := &s3.DeleteObjectInput{
		Bucket: aws.String(k.bucketName),
		Key:    aws.String(key),
	}

	_, err := k.client.DeleteObject(deleteInput)

	return err
}

//// Put payload in storage with overwrite
//func (s *S3Storage) Put(key string, payload []byte) error {
//	key = alterKey(key)
//
//	// Archive payload
//	var b bytes.Buffer
//	w := gzip.NewWriter(&b)
//	_, err := w.Write(payload)
//	if err != nil {
//		return err
//	}
//	err = w.Close()
//	if err != nil {
//		return err
//	}
//
//	// Prepare object input request
//	params := &s3.PutObjectInput{
//		Bucket: aws.String(s.BucketName),
//		Key:    aws.String(key),
//		Body:   bytes.NewReader(b.Bytes()),
//	}
//
//	for errorCounter := 0; errorCounter < 3; errorCounter++ {
//		_, err = s.Client.PutObject(params)
//		if err == nil {
//			break
//		}
//		fmt.Printf("Error writing %v: %v", key, err)
//	}
//	return err
//}
//
//// Get payload from storage, nil if not found
//func (s *S3Storage) Get(key string) ([]byte, error) {
//	key = alterKey(key)
//
//	params := &s3.GetObjectInput{
//		Bucket: aws.String(s.BucketName),
//		Key:    aws.String(key),
//	}
//	resp, err := s.Client.GetObject(params)
//
//	if err != nil {
//		return nil, err
//	}
//
//	// Decompress read data
//	r, err := gzip.NewReader(resp.Body)
//	if err != nil {
//		return nil, err
//	}
//
//	result, err := ioutil.ReadAll(r)
//	if err != nil {
//		return nil, err
//	}
//
//	err = r.Close()
//	if err != nil {
//		return nil, err
//	}
//
//	return result, nil
//}
