package sqs_messenger

import (
	"encoding/json"

	"code.justin.tv/cb/kinesis_processor/adapters/helper"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sqs"
	"github.com/aws/aws-sdk-go/service/sqs/sqsiface"
)

const (
	MessageVisibilityTimeout = 10 * 60 // 10 minutes
	NumberOfMessagesRead     = 5       // max 10
)

// sqsMessenger is SQS client.
type sqsMessenger struct {
	client   sqsiface.SQSAPI
	queueURL string
}

// NewClient create instance
func NewClient(env string, region string, queue string) SQSMessenger {
	creds := helper.NewCredentials(env, region)
	awsConfig := &aws.Config{
		Credentials: creds,
		Region:      aws.String(region),
	}

	sqsClient := sqs.New(session.New(awsConfig))

	params := &sqs.GetQueueUrlInput{
		QueueName: aws.String(queue), // Required
	}

	resp, err := sqsClient.GetQueueUrl(params)
	if err != nil {
		panic(err)
	}

	return &sqsMessenger{
		client:   sqsClient,
		queueURL: *resp.QueueUrl,
	}
}

// ReadMessages up to 5 messages from a queue
func (c *sqsMessenger) ReadMessages() ([]*Message, error) {
	params := &sqs.ReceiveMessageInput{
		QueueUrl:            aws.String(c.queueURL),
		MaxNumberOfMessages: aws.Int64(NumberOfMessagesRead),
		VisibilityTimeout:   aws.Int64(MessageVisibilityTimeout),
		WaitTimeSeconds:     aws.Int64(1),
	}
	resp, err := c.client.ReceiveMessage(params)

	if err != nil {
		return nil, err
	}
	result := make([]*Message, len(resp.Messages))
	for i, entry := range resp.Messages {
		message := &Message{}
		err := json.Unmarshal([]byte(*entry.Body), message)
		if err != nil {
			return nil, err
		}
		message.SqsMessage = entry
		result[i] = message
	}

	return result, nil
}

// DeleteMessage deletes message that were recieved from the queue
func (c *sqsMessenger) DeleteMessage(message *Message) error {

	params := &sqs.DeleteMessageInput{
		ReceiptHandle: aws.String(*message.SqsMessage.ReceiptHandle),
		QueueUrl:      aws.String(c.queueURL), // Required
	}

	_, err := c.client.DeleteMessage(params)

	return err
}

// DeleteMessages Deletes messages that were recieved from the queue
func (c *sqsMessenger) DeleteMessages(messages []*Message) error {
	entries := make([]*sqs.DeleteMessageBatchRequestEntry, len(messages), len(messages))

	for i := 0; i < len(messages); i++ {
		entries[i] = &sqs.DeleteMessageBatchRequestEntry{
			Id:            aws.String(*messages[i].SqsMessage.MessageId),
			ReceiptHandle: aws.String(*messages[i].SqsMessage.ReceiptHandle),
		}
	}

	params := &sqs.DeleteMessageBatchInput{
		Entries:  entries,
		QueueUrl: aws.String(c.queueURL), // Required
	}

	_, err := c.client.DeleteMessageBatch(params)

	return err
}
