package config

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"os"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/credentials/ec2rolecreds"
	"github.com/aws/aws-sdk-go/aws/session"
	log "github.com/sirupsen/logrus"
)

// Config contains environment-specific configuration settings for the
// application, loaded from command-line flags, environment variables,
// and config JSON.
type Config struct {
	// AWSRegion is the Amazon Web Services region name.
	AWSRegion string

	// BindAddress is the bind address port for the service.
	BindAddress string

	// Redis cache address of chat activity
	ChatActivityRedisHost string

	// Environment is the running environment determined by the environment
	// variable "ENV".
	Environment string

	// HostName is the application host name provided by the kernel.
	HostName string

	// MaxWorkerCount is the maximum number of workerpool to spawn.
	MaxWorkerCount int

	// RDSAddress is the TCP address (host and port) to the AWS RDS database
	// cluster.
	RDSAddress string

	// RDSDatabaseName is the name of the AWS RDS database cluster database.
	RDSDatabaseName string

	// RDSUsername is the username of the AWS RDS database cluster.
	RDSUsername string

	// RDSPassword is the password of the AWS RDS database cluster.
	RDSPassword string

	// RollbarToken is used to setup logging to Rollbar via rollrus package.
	RollbarToken string

	// SQSName is the name of the AWS SQS queue.
	SQSName string

	S3BucketsByEvent struct {
		Misc      string `json:"misc"`
		VideoPlay string `json:"video-play"`
	} `json:"firehoseS3Bucket"`

	// Event this specializes in
	Event string `json:",omitempty"`

	// S3 bucket for Kinesis Stream.
	FirehoseS3Bucket string `json:",omitempty"`

	// S3 bucket for Kinesis Stream Prefix.
	FirehoseS3Prefix string

	// SQS name for Session UP queue.
	SQSSessionUp string

	// SQS name for Session DOWN queue.
	SQSSessionDown string
}

// LoadConfig returns environment-specific configuration settings.
func LoadConfig() *Config {
	env := getEnv()
	event := getEvent()

	log.Info(fmt.Sprintf("Loading configurations for %s...", env))

	config, err := unmarshalJSON(env, event)
	if err != nil {
		panic(err)
	}

	config.HostName = getHostName()

	return config
}

// getEnv is a helper function to return a string of the environment variable
// "ENV".
func getEnv() string {
	env := os.Getenv("ENV")
	if env == "production" || env == "staging" || env == "test" {
		return env
	}

	return "development"
}

// getEvent is a helper function to return a string of the event name
// this processor will specialize in
func getEvent() string {
	env := os.Getenv("EVENT")
	if env == "video-play" {
		return env
	}

	return "misc"
}

// unmarshalJSON opens and reads a config JSON file specified by the
// environment string, and sets the configuration strings to an instance
// of the Config object.
func unmarshalJSON(env string, event string) (*Config, error) {
	fileName := fmt.Sprintf("./config/%s.json", env)

	file, err := ioutil.ReadFile(fileName)
	if err != nil {
		return nil, err
	}

	config := &Config{}
	err = json.Unmarshal(file, &config)
	if err != nil {
		return nil, err
	}

	config.Event = event

	switch event {
	case "video-play":
		config.FirehoseS3Bucket = config.S3BucketsByEvent.VideoPlay
	case "misc":
		config.FirehoseS3Bucket = config.S3BucketsByEvent.Misc
	}

	return config, nil
}

// getHostName returns either the kernel host name or otherwise "unknown".
func getHostName() string {
	hostName, err := os.Hostname()
	if err != nil {
		return "unknown"
	}

	return hostName
}

// DBDataSource returns a formatted connection string
func (conf *Config) DBDataSource() string {
	dbMode := ""
	if conf.Environment == "development" {
		dbMode = "sslmode=disable"
	}
	return fmt.Sprintf(
		//"%s:%s@tcp(%s)/%s?parseTime=true",
		"postgres://%s:%s@%s/%s?%s",
		conf.RDSUsername,
		conf.RDSPassword,
		conf.RDSAddress,
		conf.RDSDatabaseName,
		dbMode,
	)
}

// NewAWSCredentials return AWS Credentials object with keys
func (c *Config) NewAWSCredentials() *credentials.Credentials {
	if c.Environment == "development" {
		return credentials.NewSharedCredentials("", "twitch-web-dev")
	}

	return ec2rolecreds.NewCredentials(
		session.New(&aws.Config{
			Region: aws.String(c.AWSRegion),
		}),
	)
}
