provider "aws" {
  region = "us-west-2"
}

# Name of the Kinesis Firehose stream to be created: `spade-downstream-prod-{id}`.
variable "misc-stream-name" {
  type = "string"
  default = "cb-analytics"
}

variable "video-play-stream-name" {
  type = "string"
  default = "spade-downstream-prod-cb-analytics-video-play"
}

# Name of the s3 bucket to write data to (created here).
# If you want to use a pre-existing bucket, you can delete the aws_s3_bucket
# resource below. Make sure to then set an expiration policy on the prefix.
variable "misc-bucket-name" {
  type = "string"
  default = "cb-kinesis-s3"
}

variable "video-play-bucket-name" {
  type = "string"
  default = "cb-kinesis-s3-video-play"
}

# Prefix to write data into (including a trailing slash unless it's empty).
variable "bucket-prefix" {
  type = "string"
  default = "incomming/"
}

# Number of days to keep data in the receiving s3 bucket.
# Delete this variable if using your own s3 bucket.
variable "expiration-days" {
  type = "string"
  default = "10"
}

# AWS account ID. Available with `aws sts get-caller-identity`.
variable "account-id" {}

# Pre-existing SNS topic ARN in same account to notify about nearing capacity.
variable "our-sns-topic" {
  type = "string"
  default = "cb-analytics-kinesis-alerts"
}

# Firehose capacity can only be increased with a support ticket. If you increase
# your capacity, increase this number proportionately to get useful alarms below.
variable "shard-count" {
  default = 5
}

variable "scieng-sns-topic" {
  default = "arn:aws:sns:us-west-2:673385534282:spade-downstream-prod-alert"
}

resource "aws_s3_bucket" "sink" {
  bucket = "${var.misc-bucket-name}"
  acl    = "private"

  lifecycle_rule {
    prefix  = "${var.bucket-prefix}"
    enabled = true

    expiration = {
      days = "${var.expiration-days}"
    }
  }
}

resource "aws_s3_bucket" "video-play" {
  bucket = "${var.video-play-bucket-name}"
  acl    = "private"

  lifecycle_rule {
    prefix  = "${var.bucket-prefix}"
    enabled = true

    expiration = {
      days = "${var.expiration-days}"
    }
  }
}

resource "aws_kinesis_firehose_delivery_stream" "misc-stream" {
  name        = "${var.misc-stream-name}"
  destination = "s3"

  s3_configuration {
    role_arn = "${aws_iam_role.firehose.arn}"
    bucket_arn = "${aws_s3_bucket.sink.arn}"
    prefix = "${var.bucket-prefix}"
    buffer_interval = 60
  }
}

resource "aws_kinesis_firehose_delivery_stream" "video-play-stream" {
  name        = "${var.video-play-stream-name}"
  destination = "s3"

  s3_configuration {
    role_arn = "${aws_iam_role.firehose.arn}"
    bucket_arn = "${aws_s3_bucket.video-play.arn}"
    prefix = "${var.bucket-prefix}"
    buffer_interval = 60
  }
}

resource "aws_cloudwatch_metric_alarm" "misc-alarm" {
  alarm_name          = "${var.misc-stream-name}-too-many-bytes"
  alarm_description   = "Too many bytes sent to ${var.misc-stream-name}"
  comparison_operator = "GreaterThanThreshold"
  evaluation_periods  = "2"
  metric_name         = "IncomingBytes"
  namespace           = "AWS/Firehose"
  period              = "60"
  statistic           = "Sum"
  threshold           = "${0.85 * var.shard-count * 1000000 * 60}"

  dimensions {
    DeliveryStreamName = "${aws_kinesis_firehose_delivery_stream.misc-stream.name}"
  }

  alarm_actions             = ["${var.scieng-sns-topic}", "${aws_sns_topic.sns_alert_topic.id}"]
  ok_actions                = ["${var.scieng-sns-topic}", "${aws_sns_topic.sns_alert_topic.id}"]
  insufficient_data_actions = ["${var.scieng-sns-topic}", "${aws_sns_topic.sns_alert_topic.id}"]
}

resource "aws_cloudwatch_metric_alarm" "video-play-alarm" {
  alarm_name          = "${var.video-play-stream-name}-too-many-bytes"
  alarm_description   = "Too many bytes sent to ${var.video-play-stream-name}"
  comparison_operator = "GreaterThanThreshold"
  evaluation_periods  = "2"
  metric_name         = "IncomingBytes"
  namespace           = "AWS/Firehose"
  period              = "60"
  statistic           = "Sum"
  threshold           = "${0.85 * var.shard-count * 1000000 * 60}"

  dimensions {
    DeliveryStreamName = "${aws_kinesis_firehose_delivery_stream.video-play-stream.name}"
  }

  alarm_actions             = ["${var.scieng-sns-topic}", "${aws_sns_topic.sns_alert_topic.id}"]
  ok_actions                = ["${var.scieng-sns-topic}", "${aws_sns_topic.sns_alert_topic.id}"]
  insufficient_data_actions = ["${var.scieng-sns-topic}", "${aws_sns_topic.sns_alert_topic.id}"]
}

resource "aws_cloudwatch_metric_alarm" "misc-stream-records" {
  alarm_name          = "${var.misc-stream-name}-too-many-records"
  alarm_description   = "Too many records sent to ${var.misc-stream-name}"
  comparison_operator = "GreaterThanThreshold"
  evaluation_periods  = "2"
  metric_name         = "IncomingRecords"
  namespace           = "AWS/Firehose"
  period              = "60"
  statistic           = "Sum"
  threshold           = "${0.85 * var.shard-count * 1000 * 60}"

  dimensions {
    DeliveryStreamName = "${aws_kinesis_firehose_delivery_stream.misc-stream.name}"
  }

  alarm_actions             = ["${var.scieng-sns-topic}", "${aws_sns_topic.sns_alert_topic.id}"]
  ok_actions                = ["${var.scieng-sns-topic}", "${aws_sns_topic.sns_alert_topic.id}"]
  insufficient_data_actions = ["${var.scieng-sns-topic}", "${aws_sns_topic.sns_alert_topic.id}"]
}

resource "aws_cloudwatch_metric_alarm" "video-play-stream-records" {
  alarm_name          = "${var.video-play-stream-name}-too-many-records"
  alarm_description   = "Too many records sent to ${var.video-play-stream-name}"
  comparison_operator = "GreaterThanThreshold"
  evaluation_periods  = "2"
  metric_name         = "IncomingRecords"
  namespace           = "AWS/Firehose"
  period              = "60"
  statistic           = "Sum"
  threshold           = "${0.85 * var.shard-count * 1000 * 60}"

  dimensions {
    DeliveryStreamName = "${aws_kinesis_firehose_delivery_stream.video-play-stream.name}"
  }

  alarm_actions             = ["${var.scieng-sns-topic}", "${aws_sns_topic.sns_alert_topic.id}"]
  ok_actions                = ["${var.scieng-sns-topic}", "${aws_sns_topic.sns_alert_topic.id}"]
  insufficient_data_actions = ["${var.scieng-sns-topic}", "${aws_sns_topic.sns_alert_topic.id}"]
}

resource "aws_iam_role" "spade-putter" {
  name = "cb-analytics-putter"
  path = "/"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Principal": {
        "AWS": [
          "arn:aws:iam::673385534282:root"
        ]
      },
      "Effect": "Allow"
    }
  ]
}
EOF
}

resource "aws_iam_role_policy" "spade-putter" {
  name = "cb-analytics-putter"
  role = "${aws_iam_role.spade-putter.id}"

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": [
        "firehose:PutRecordBatch"
      ],
      "Effect": "Allow",
      "Resource": [
        "${aws_kinesis_firehose_delivery_stream.misc-stream.arn}",
        "${aws_kinesis_firehose_delivery_stream.video-play-stream.arn}"
      ]
    }
  ]
}
EOF
}

resource "aws_iam_role" "firehose" {
  name = "cb-analytics-firehose"
  path = "/"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Principal": {
        "Service": "firehose.amazonaws.com"
      },
      "Effect": "Allow",
      "Condition": {
        "StringEquals": {
          "sts:ExternalId":"${var.account-id}"
        }
      }
    }
  ]
}
EOF
}

resource "aws_iam_role_policy" "firehose" {
  name = "cb-analytics-firehose"
  role = "${aws_iam_role.firehose.id}"

  policy = <<EOF
{
"Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Action": [
        "s3:AbortMultipartUpload",
        "s3:GetBucketLocation",
        "s3:GetObject",
        "s3:ListBucket",
        "s3:ListBucketMultipartUploads",
        "s3:PutObject"
      ],
      "Resource": [
        "arn:aws:s3:::${var.misc-bucket-name}",
        "arn:aws:s3:::${var.misc-bucket-name}/${var.bucket-prefix}*",
        "arn:aws:s3:::${var.video-play-bucket-name}",
        "arn:aws:s3:::${var.video-play-bucket-name}/${var.bucket-prefix}*"
      ]
    }
  ]
}
EOF
}

output "iam-role" {
  value = "${aws_iam_role.spade-putter.arn}"
}

resource "aws_sns_topic" "sns_alert_topic" {
  name = "${var.our-sns-topic}"
}
