﻿using Amazon;
using Amazon.S3;
using Amazon.S3.Model;
using GlueFactory.Models;
using Newtonsoft.Json;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.IO.Compression;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading;

namespace GlueFactory.Helpers
{
    public static class S3Helper
    {
        private static AmazonS3Client s3Client = new AmazonS3Client(RegionEndpoint.USWest2);

        static S3Helper()
        {

        }

        public static AmazonS3Client GetActiveClient()
        {
            return s3Client;
        }

        public static ListObjectsResponse[] GetAllFilesAtPath(string bucket, string prefix, AmazonS3Client replacementClient = null, int maxKeys = 1000)
        {
            var activeS3Client = replacementClient ?? s3Client;
            ListObjectsRequest request = new ListObjectsRequest()
            {
                BucketName = bucket,
                MaxKeys = 1000,
                Prefix = prefix
            };
            List<ListObjectsResponse> responses = new List<ListObjectsResponse>();
            do
            {
                var responseTask = activeS3Client.ListObjectsAsync(request);
                responseTask.Wait();
                ListObjectsResponse response = responseTask.Result;
                if (response != null)
                {
                    responses.Add(response);
                    if (response.IsTruncated)
                    {
                        request.Marker = response.NextMarker;
                    }
                    else
                    {
                        request = null;
                    }
                }
                else
                {
                    request = null;
                }
            } while (request != null);
            return responses.ToArray();
        }

        public static void WriteStringToS3(string data, string bucket, string keypath, bool compress = true, string kmsKeyID = null, ServerSideEncryptionMethod encryptionMethod = null)
        {
            if (compress)
            {
                using (var stream = new MemoryStream())
                {
                    using (var gZipStream = new GZipStream(stream, CompressionMode.Compress, true))
                    {
                        using (TextWriter writer = new StreamWriter(gZipStream, Encoding.UTF8, 1024, leaveOpen: true))
                        {
                            writer.Write(data);
                        }
                    }
                    stream.Position = 0;
                    UploadToS3(stream, bucket, keypath, kmsKeyID, encryptionMethod);
                    stream.Close();
                }
            }
            else
            {
                using (var stream = new MemoryStream())
                {
                    using (TextWriter writer = new StreamWriter(stream, Encoding.UTF8, 1024, leaveOpen: true))
                    {
                        writer.Write(data);
                    }
                    stream.Position = 0;
                    UploadToS3(stream, bucket, keypath, kmsKeyID, encryptionMethod);
                    stream.Close();
                }
            }
        }

        public static void UploadToS3(string inputFile, string bucket, string path, string kmsKeyID = null, ServerSideEncryptionMethod encryptionMethod = null)
        {
            FileInfo finfo = new FileInfo(inputFile);

            if (finfo.Length > 4242880)
            {
                List<UploadPartResponse> uploadResponses = new List<UploadPartResponse>();
                List<PartETag> partETags = new List<PartETag>();

                InitiateMultipartUploadRequest initiateRequest = new InitiateMultipartUploadRequest
                {
                    BucketName = bucket,
                    Key = path
                };
                if (kmsKeyID != null)
                {
                    initiateRequest.ServerSideEncryptionMethod = encryptionMethod;
                    initiateRequest.ServerSideEncryptionKeyManagementServiceKeyId = kmsKeyID;
                }

                var initResponseTask = s3Client.InitiateMultipartUploadAsync(initiateRequest);
                initResponseTask.Wait();
                InitiateMultipartUploadResponse initResponse = initResponseTask.Result;

                long contentLength = finfo.Length;
                long partSize = 52428800; // 50 MB

                try
                {
                    long filePosition = 0;
                    for (int i = 1; filePosition < contentLength; i++)
                    {

                        UploadPartRequest uploadRequest = new UploadPartRequest
                        {
                            BucketName = bucket,
                            Key = path,
                            UploadId = initResponse.UploadId,
                            PartNumber = i,
                            PartSize = partSize,
                            FilePosition = filePosition,
                            FilePath = inputFile
                        };

                        var partResponseTask = s3Client.UploadPartAsync(uploadRequest);
                        partResponseTask.Wait();
                        var partResponse = partResponseTask.Result;
                        uploadResponses.Add(partResponse);

                        PartETag petag = new PartETag(partResponse.PartNumber, partResponse.ETag);
                        partETags.Add(petag);

                        filePosition += partSize;
                    }

                    CompleteMultipartUploadRequest completeRequest = new CompleteMultipartUploadRequest
                    {
                        BucketName = bucket,
                        Key = path,
                        UploadId = initResponse.UploadId,
                        PartETags = partETags
                    };

                    var completeUploadResponseTask = s3Client.CompleteMultipartUploadAsync(completeRequest);
                    completeUploadResponseTask.Wait();
                    CompleteMultipartUploadResponse completeUploadResponse = completeUploadResponseTask.Result;
                }
                catch (Exception)
                {
                    AbortMultipartUploadRequest abortMPURequest = new AbortMultipartUploadRequest
                    {
                        BucketName = bucket,
                        Key = path,
                        UploadId = initResponse.UploadId
                    };
                    var abortTask = s3Client.AbortMultipartUploadAsync(abortMPURequest);
                    abortTask.Wait();
                }
            }
            else
            {
                PutObjectRequest request = new PutObjectRequest()
                {
                    BucketName = bucket,
                    FilePath = inputFile,
                    Key = path
                };
                if (kmsKeyID != null)
                {
                    request.ServerSideEncryptionMethod = ServerSideEncryptionMethod.AWSKMS;
                    request.ServerSideEncryptionKeyManagementServiceKeyId = kmsKeyID;
                }
                var responseTask = s3Client.PutObjectAsync(request);
                responseTask.Wait();
                var response = responseTask.Result;
            }
        }

        public static void UploadToS3(Stream inputStream, string bucket, string path, string kmsKeyID = null, ServerSideEncryptionMethod encryptionMethod = null, bool availableToPublic = false)
        {
            inputStream.Seek(0, SeekOrigin.Begin);

            if (inputStream.Length > 4242880)
            {
                List<UploadPartResponse> uploadResponses = new List<UploadPartResponse>();
                List<PartETag> partETags = new List<PartETag>();

                InitiateMultipartUploadRequest initiateRequest = new InitiateMultipartUploadRequest
                {
                    BucketName = bucket,
                    Key = path
                };
                if (kmsKeyID != null)
                {
                    initiateRequest.ServerSideEncryptionMethod = encryptionMethod ?? ServerSideEncryptionMethod.AWSKMS;
                    initiateRequest.ServerSideEncryptionKeyManagementServiceKeyId = kmsKeyID;
                }

                var initResponseTask = s3Client.InitiateMultipartUploadAsync(initiateRequest);
                initResponseTask.Wait();
                InitiateMultipartUploadResponse initResponse = initResponseTask.Result;

                long contentLength = inputStream.Length;
                long partSize = 52428800; // 50 MB

                try
                {
                    long filePosition = 0;
                    for (int i = 1; filePosition < contentLength; i++)
                    {

                        UploadPartRequest uploadRequest = new UploadPartRequest
                        {
                            BucketName = bucket,
                            Key = path,
                            UploadId = initResponse.UploadId,
                            PartNumber = i,
                            PartSize = partSize,
                            FilePosition = filePosition,
                            InputStream = inputStream
                        };

                        var partResponseTask = s3Client.UploadPartAsync(uploadRequest);
                        partResponseTask.Wait();
                        var partResponse = partResponseTask.Result;
                        uploadResponses.Add(partResponse);

                        PartETag petag = new PartETag(partResponse.PartNumber, partResponse.ETag);
                        partETags.Add(petag);

                        filePosition += partSize;
                    }

                    CompleteMultipartUploadRequest completeRequest = new CompleteMultipartUploadRequest
                    {
                        BucketName = bucket,
                        Key = path,
                        UploadId = initResponse.UploadId,
                        PartETags = partETags
                    };

                    var completeUploadResponseTask = s3Client.CompleteMultipartUploadAsync(completeRequest);
                    completeUploadResponseTask.Wait();
                    CompleteMultipartUploadResponse completeUploadResponse = completeUploadResponseTask.Result;
                }
                catch (Exception)
                {
                    AbortMultipartUploadRequest abortMPURequest = new AbortMultipartUploadRequest
                    {
                        BucketName = bucket,
                        Key = path,
                        UploadId = initResponse.UploadId
                    };
                    var abortTask = s3Client.AbortMultipartUploadAsync(abortMPURequest);
                    abortTask.Wait();
                }
            }
            else
            {
                PutObjectRequest request = new PutObjectRequest()
                {
                    BucketName = bucket,
                    InputStream = inputStream,
                    Key = path
                };
                if (kmsKeyID != null)
                {
                    request.ServerSideEncryptionMethod = ServerSideEncryptionMethod.AWSKMS;
                    request.ServerSideEncryptionKeyManagementServiceKeyId = kmsKeyID;
                }
                var responseTask = s3Client.PutObjectAsync(request);
                responseTask.Wait();
                var response = responseTask.Result;
            }
        }

        /// <summary>
        /// Returns True (Exists) | False (Not Exists) | Null (Exception)
        /// </summary>
        public static bool? ExistsInS3(string bucket, string path, AmazonS3Client replacementClient = null)
        {
            bool? uploaded = null;
            try
            {
                var activeS3Client = replacementClient ?? s3Client;
                GetObjectMetadataRequest existRequest = new GetObjectMetadataRequest();
                existRequest.BucketName = bucket;
                existRequest.Key = path;
                var existResponseTask = activeS3Client.GetObjectMetadataAsync(existRequest);
                existResponseTask.Wait();
                var existResponse = existResponseTask.Result;
                uploaded = existResponse.ContentLength != 0;
            }
            catch (Exception ex)
            {
                if (ex.InnerException != null && ex.InnerException.GetType() == typeof(AmazonS3Exception))
                {
                    if (ex.InnerException.Message.Contains("NotFound"))
                    {
                        uploaded = false;
                    }
                }
                else
                {
                    Console.WriteLine("S3 Error: " + ex.ToString());
                }
            }
            return uploaded;
        }

        public static DateTime? LastUpdatedInS3(string bucket, string path, AmazonS3Client replacementClient = null)
        {
            DateTime? lastUpdated = null;
            try
            {
                var activeS3Client = replacementClient ?? s3Client;
                GetObjectMetadataRequest existRequest = new GetObjectMetadataRequest();
                existRequest.BucketName = bucket;
                existRequest.Key = path;
                var existResponseTask = activeS3Client.GetObjectMetadataAsync(existRequest);
                existResponseTask.Wait();
                var existResponse = existResponseTask.Result;
                lastUpdated = existResponse.LastModified;
            }
            catch
            {
            }
            return lastUpdated;
        }

        public static GetObjectMetadataResponse GetFileMetaData(string bucket, string path, AmazonS3Client replacementClient = null)
        {
            try
            {
                var activeS3Client = replacementClient ?? s3Client;
                GetObjectMetadataRequest existRequest = new GetObjectMetadataRequest();
                existRequest.BucketName = bucket;
                existRequest.Key = path;
                var existResponseTask = activeS3Client.GetObjectMetadataAsync(existRequest);
                existResponseTask.Wait();
                var existResponse = existResponseTask.Result;
                return existResponse;
            }
            catch (Exception)
            {
                if (replacementClient != null)
                {
                    throw;
                }
                return null;
            }
        }

        public static void DownloadFromS3WithFixedSizeRange(string bucket, string path, string localWritePath, AmazonS3Client replacementClient = null, int batchSize = 500000000)
        {
            var activeS3Client = replacementClient ?? s3Client;

            var metadata = GetFileMetaData(bucket, path, replacementClient);
            long currentPos = 0;
            using (var filestream = new FileStream(localWritePath, FileMode.Create, FileAccess.ReadWrite, FileShare.ReadWrite))
            {
                do
                {
                    try
                    {
                        var mod = currentPos + batchSize;
                        var remainder = metadata.ContentLength - currentPos;
                        var endPos = mod < metadata.ContentLength
                            ? mod
                            : metadata.ContentLength;
#if CONFIG_DEVELOPMENT
                        Console.WriteLine($"Downloading {bucket}/{path}, Byte Range: {currentPos.ToString("N0")}-{endPos.ToString("N0")}, Remaining: {remainder.ToString("N0")}");
                        Debug.Print($"Downloading {bucket}/{path}, Byte Range: {currentPos.ToString("N0")}-{endPos.ToString("N0")}, Remaining: {remainder.ToString("N0")}");
#endif
                        var request = new GetObjectRequest()
                        {
                            BucketName = bucket,
                            Key = path,
                            ByteRange = new ByteRange(currentPos, endPos)
                        };
                        var responseTask = activeS3Client.GetObjectAsync(request);
                        responseTask.Wait();
                        using (var response = responseTask.Result)
                        {
                            using (var stream = response.ResponseStream)
                            {
                                var data = ReadFully(stream);
                                filestream.Write(data, 0, data.Length);
                                filestream.Flush();
                            }
                        }
                        currentPos = endPos + 1;
                    }
                    catch (Exception ex)
                    {
#if CONFIG_DEVELOPMENT
                        Console.WriteLine($"S3Helper DownloadFromS3WithFiedSizeRange Error: {ex}");
#endif
                    }
                } while (currentPos < metadata.ContentLength);
                filestream.Flush();
            }
        }

        public static byte[] DownloadFromS3ExplicitByteRange(string bucket, string path, long startByte, long endByte, AmazonS3Client replacementClient = null)
        {
            var activeS3Client = replacementClient ?? s3Client;

            try
            {
                var request = new GetObjectRequest()
                {
                    BucketName = bucket,
                    Key = path,
                    ByteRange = new ByteRange(startByte, endByte)
                };
                var responseTask = activeS3Client.GetObjectAsync(request);
                responseTask.Wait();
                using (var response = responseTask.Result)
                {
                    using (var stream = response.ResponseStream)
                    {
                        var data = ReadFully(stream);
                        return data;
                    }
                }
            }
            catch (Exception ex)
            {
#if CONFIG_DEVELOPMENT
                Console.WriteLine($"S3Helper DownloadFromS3WithFiedSizeRange Error: {ex}");
#endif
            }
            return null;
        }

        public static DeleteObjectResponse DeleteFromS3(string bucket, string path)
        {
            DeleteObjectRequest request = new DeleteObjectRequest
            {
                BucketName = bucket,
                Key = path,
            };
            var responseTask = s3Client.DeleteObjectAsync(request);
            responseTask.Wait();
            DeleteObjectResponse response = responseTask.Result;
            return response;
        }

        public static byte[] DownloadFromS3(string bucket, string path, bool isCompressed = false, AmazonS3Client replacementClient = null)
        {
            try
            {
                var activeS3Client = replacementClient ?? s3Client;

                var request = new GetObjectRequest()
                {
                    BucketName = bucket,
                    Key = path,
                };

                var responseTask = activeS3Client.GetObjectAsync(request);
                responseTask.Wait();
                using (var response = responseTask.Result)
                {
                    using (var stream = response.ResponseStream)
                    {
                        var data = ReadFully(stream, isCompressed);
                        return data;
                    }
                }
            }
            catch (Exception ex)
            {
                Debug.Print(ex.ToString());
                return null;
            }
        }

        public static byte[] ReadFully(Stream input, bool isCompressed = false)
        {
            byte[] buffer = new byte[16 * 1024];
            if (isCompressed)
            {
                using (var gstream = input)
                {
                    using (var stream = new GZipStream(gstream, CompressionMode.Decompress))
                    {
                        using (MemoryStream ms = new MemoryStream())
                        {
                            int read;
                            while ((read = stream.Read(buffer, 0, buffer.Length)) > 0)
                            {
                                ms.Write(buffer, 0, read);
                            }
                            return ms.ToArray();
                        }
                    }
                }
            }
            else
            {
                using (MemoryStream ms = new MemoryStream())
                {
                    int read;
                    while ((read = input.Read(buffer, 0, buffer.Length)) > 0)
                    {
                        ms.Write(buffer, 0, read);
                    }
                    return ms.ToArray();
                }
            }
        }

        /// <summary>
        /// Gets a list of T from the query. Keep in mind: Limit queries include the header row, so limit 100 returns 99 rows of data
        /// </summary>
        public static List<T> GetCsvDataObjectList<T>(string bucket, string keypath, bool isCompressed = true, AmazonS3Client replacementClient = null, int pollTimer = 1000, int pollTimeout = 60000) where T : class
        {
            var activeS3Client = replacementClient ?? s3Client;
            bool success = false;
            bool breakcheck = false;
            var timeout = DateTime.UtcNow.AddMilliseconds(pollTimeout);
            if (pollTimeout == 0)
            {
                timeout = new DateTime(9999, 12, 31);
            }
            do
            {
                try
                {
                    var exists = S3Helper.ExistsInS3(bucket, keypath, activeS3Client) == true;
                    if (!exists)
                    {
                        if (exists == false)
                        {
                            return new List<T>();
                        }
                        else
                        {
                            return null;
                        }
                    }
                    success = true;
                    breakcheck = true;
                }
                catch (AmazonS3Exception ex)
                {
                    // This occurs when the query is fairly large, and it takes a bit to write the results
                    if (!ex.Message.Contains("NotFound"))
                    {
                        breakcheck = true;
                        throw;
                    }
                    Debug.Print($"S3 file {bucket}/{keypath} not ready. Waiting for next poll. Expiration at {timeout.ToString("yyyy-MM-dd HH:mm:ss")}");
                    Thread.Sleep(pollTimer);
                }
                catch (Exception ex)
                {
                    breakcheck = true;
                    Debug.Print(ex.ToString());
                    return null;
                }
            } while (!success && !breakcheck && DateTime.UtcNow <= timeout);

            var request = new GetObjectRequest()
            {
                BucketName = bucket,
                Key = keypath,
            };

            try
            {
                var records = new List<T>();
                var responseTask = activeS3Client.GetObjectAsync(request);
                responseTask.Wait();
                using (var response = responseTask.Result)
                {
                    if (isCompressed)
                    {
                        using (var gstream = response.ResponseStream)
                        {
                            using (var stream = new GZipStream(gstream, CompressionMode.Decompress))
                            {
                                using (TextReader reader = new StreamReader(stream, Encoding.UTF8))
                                {
                                    using (var csv = new CsvHelper.CsvReader(reader, new CsvHelper.Configuration.Configuration()
                                    {
                                        Delimiter = ",",
                                        Quote = '"',
                                        Encoding = Encoding.UTF8,
                                        BufferSize = 2048,
                                        ShouldQuote = (field, context) => true
                                    }))
                                    {
                                        csv.Read();
                                        records.AddRange(csv.GetRecords<T>());
                                    }
                                }
                            }
                        }
                    }
                    else
                    {
                        using (var stream = response.ResponseStream)
                        {
                            using (TextReader reader = new StreamReader(stream, Encoding.UTF8))
                            {
                                using (var csv = new CsvHelper.CsvReader(reader, new CsvHelper.Configuration.Configuration()
                                {
                                    Delimiter = ",",
                                    Quote = '"',
                                    Encoding = Encoding.UTF8,
                                    BufferSize = 2048,
                                    ShouldQuote = (field, context) => true
                                }))
                                {
                                    csv.Read();
                                    records.AddRange(csv.GetRecords<T>());
                                }
                            }
                        }
                    }
                }

                return records;
            }
            catch (Exception ex)
            {
                Debug.Print(ex.ToString());
                throw;
            }
        }

        public static string ReadTextFromS3(string bucket, string keypath, bool isCompressed = true, AmazonS3Client replacementClient = null)
        {
            var activeS3Client = replacementClient ?? s3Client;

            var request = new GetObjectRequest()
            {
                BucketName = bucket,
                Key = keypath
            };

            var responseTask = activeS3Client.GetObjectAsync(request);
            responseTask.Wait();
            using (var response = responseTask.Result)
            {
                string result = null;
                if (isCompressed)
                {
                    using (var gstream = response.ResponseStream)
                    {
                        using (var stream = new GZipStream(gstream, CompressionMode.Decompress))
                        {
                            using (TextReader reader = new StreamReader(stream, Encoding.UTF8))
                            {
                                result = reader.ReadToEnd();
                            }
                        }
                    }
                }
                else
                {
                    using (TextReader reader = new StreamReader(response.ResponseStream))
                    {
                        result = reader.ReadToEnd();
                    }
                }
                return result;
            }
        }

        public static string[] GetManifestUrls(string s3bucket, string keyfolderpath, ref ManifestModel data)
        {
            var contentManifests = new List<string>();
            if (data != null && data.ManifestFiles.Count > 0)
            {
                foreach (var keypath in data.ManifestFiles)
                {
                    var s3keypath = $@"{keypath}data_manifest";
                    if (S3Helper.ExistsInS3(s3bucket, s3keypath) == true)
                    {
                        var manifests = JsonConvert.DeserializeObject<UnloadEntries>(S3Helper.ReadTextFromS3(s3bucket, s3keypath, false));
                        if (manifests != null && manifests.Manifests != null && manifests.Manifests.Length > 0)
                        {
                            foreach (var manifest in manifests.Manifests)
                            {
                                if (manifest.Data.ContentLength == 0)
                                {
                                    continue;
                                }

                                if (Constants.UnloadManifestRegex.IsMatch(manifest.Url))
                                {
                                    foreach (Match match in Constants.UnloadManifestRegex.Matches(manifest.Url))
                                    {
                                        if (match.Groups.Count > 0)
                                        {
                                            contentManifests.Add(match.Groups[1].Captures[0].Value);
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
            return contentManifests.ToArray();
        }

        // Process a list of manifest files, calling the callback action after reading each file
        public static void ProcessManifestContentsByBatch<T>(string s3bucket, string manifestpath, ref ManifestModel data, Action<T[]> processBatch)
        {
            throw new NotImplementedException();
        }

        // Return a list of data from all the manifest files
        public static List<T> GetCsvManifestContents<T>(string s3bucket, string manifestpath, ref ManifestModel data)
        {
            throw new NotImplementedException();
        }
    }
}
