﻿using GlueFactory.Helpers;
using GlueFactory.Models;
using Newtonsoft.Json;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;

namespace GlueFactory
{
    public class Program
    {
        private const string configPrefixPostfixPattern = "(.*?)(table_config = {(?:.*EOF\\s+}))(.*)";
        private static Regex configPrefixPostfixRegex = new Regex(configPrefixPostfixPattern, RegexOptions.Compiled | RegexOptions.Singleline | RegexOptions.IgnoreCase);
        private const string tableNamePattern = "\"(.*?)\"(?:.*<<EOF?)";
        private static Regex tableNameRegex = new Regex(tableNamePattern, RegexOptions.Compiled | RegexOptions.Singleline | RegexOptions.IgnoreCase);
        private static MySqlModel activeConnection = null;
        /// <summary>
        /// "sensitivity": {"type": "string", "enum": ["none", "userid", "ip", "sessionid", "otherid"]}
        /// </summary>
        private static Dictionary<string, string> sensitiveColumnNames = new Dictionary<string, string>()
        {
            { "channel_id", "userid"},
            { "user_id", "userid" },
            { "account_manager_email", "otherid" },
            { "account_manager_first_name", "otherid" },
            { "account_manager_last_name", "otherid" },
            { "account_manager_ldap_name", "otherid" },
            { "approving_manager_amazon_id", "otherid" },
            { "approving_manager_name", "otherid" },
            { "premium_content_creator_name", "otherid" },
            { "stream_login", "otherid" },
        };

        public static void Main(string[] args)
        {
            try
            {
                Console.WriteLine($@"Please enter project you wish to use a template for");
                var originalTableConfig = new StringBuilder();
                var originalTableData = new Dictionary<string, TableModel>();
                var newTableConfig = new StringBuilder();
                var newTableData = new Dictionary<string, TableModel>();
                var auditlog = new StringBuilder();
                var auditlogWritePath = $@"{Path.GetFullPath($@"{Environment.CurrentDirectory}/audit/{DateTime.UtcNow.ToString("yyyy-MM-dd_HH_mm_ss")}_audit_log.dat")}";
                var templateOutputFile = string.Empty;
                var templateOutputPath = string.Empty;
                var templatePrefix = string.Empty;
                var templateTables = string.Empty;
                var templatePostfix = string.Empty;

                var input = Console.ReadLine();

                string workingFilePath = string.Empty;
                if (!string.IsNullOrWhiteSpace(input))
                {
                    switch (input.ToLower())
                    {
                        case "amp":
                        {
                            activeConnection = JsonConvert.DeserializeObject<MySqlModel>(S3Helper.ReadTextFromS3("resonance-configuration", $"Production/credentials/mysql/resonance.json.gz", true, null));
                            break;
                        }
                        case "atlas":
                        {
                            activeConnection = JsonConvert.DeserializeObject<MySqlModel>(S3Helper.ReadTextFromS3("resonance-configuration", $"Production/credentials/mysql/atlas.json.gz", true, null));
                            templateOutputFile = "rds-atlas.tf";
                            workingFilePath = $@"Templates/AtlasTemplates/{templateOutputFile}";
                            templateOutputPath = $@"C:\Projects\S3GlueAtlas\";
                            break;
                        }
                        default:
                        {
                            Console.WriteLine("Unknown project.");
                            break;
                        }
                    }
                }

                if (!string.IsNullOrWhiteSpace(workingFilePath))
                {
                    //FOR TESTING var data = File.ReadAllText(Path.GetFullPath($@"{Environment.CurrentDirectory}/{workingFilePath}"));
                    //FOR PRODUCTION - Make sure you've pulled latest for the glue project!
                    var data = File.ReadAllText(Path.GetFullPath($@"{templateOutputPath}{templateOutputFile}"));
                    originalTableConfig.AppendLine(data);
                    if (configPrefixPostfixRegex.IsMatch(data))
                    {
                        var configMatches = configPrefixPostfixRegex.Match(data);
                        templatePrefix = configMatches.Groups[1].Value;
                        templateTables = configMatches.Groups[2].Value;
                        templatePostfix = configMatches.Groups[3].Value;

                        var workString = templateTables;
                        // Process templateTables for all tables                            
                        bool didWork = false;

                        do
                        {
                            didWork = false;
                            var startIndicatorString = "<<EOF";
                            var endIndicatorString = "EOF";
                            var firstIndex = workString.IndexOf(startIndicatorString);
                            if(firstIndex > 0)
                            {
                                var prefixData = workString.Substring(0, firstIndex+ startIndicatorString.Length);
                                var postfixData = workString.Substring(firstIndex + startIndicatorString.Length);

                                if (tableNameRegex.IsMatch(prefixData))
                                {
                                    var tableName = tableNameRegex.Match(prefixData).Groups[1].Value;
                                    var secondIndex = postfixData.IndexOf(endIndicatorString);
                                    if (secondIndex > 0)
                                    {
                                        try
                                        {
                                            var tableContext = JsonConvert.DeserializeObject<TableModel>(postfixData.Substring(0, secondIndex).Trim());
                                            
                                            workString = postfixData.Substring(secondIndex + endIndicatorString.Length);
                                            didWork = true;
                                            originalTableData.Add(tableName, tableContext);
                                        }
                                        catch (Exception ex)
                                        {
                                            Console.WriteLine(ex);
                                            didWork = false;
                                        }
                                    }
                                }
                            }
                        } while (workString.IndexOf("<<EOF") > 0 && didWork);
                    }
                    else
                    {
                        Console.WriteLine($@"Invalid config format.");
                        Console.ReadLine();
                        return;
                    }
                }

                if(string.IsNullOrWhiteSpace(templatePrefix) || string.IsNullOrWhiteSpace(templatePostfix))
                {
                    Console.WriteLine($@"Prefix/Postfix blank. This will require debugging. Does your file template match the regex?");
                    Console.ReadLine();
                    return;
                }

                try
                {
                    var fullDbOutput = new Dictionary<string, List<TableGlueFormatModel>>();

                    using (var conn = DBHelper.GetConnection(activeConnection, true))
                    {
                        using (var command = conn.GetCommand())
                        {
                            command.CommandText =
                            @"
                                select
                                    table_name, column_name, data_type, max_length, formatted_field_type,
                                    concat('{""name"": ""',  column_name, '"", ""type"": ""', formatted_field_type, '""},') as formatted_string
                                from
                                (
                                    select
                                        table_name,
                                        column_name,
                                        data_type,
                                        coalesce(character_maximum_length, numeric_precision) as max_length,
                                        case
                                            when data_type = 'int' then 'int'
                                            when data_type = 'varchar' then 'string'
                                            when data_type = 'text' then 'string'
                                            when data_type = 'mediumtext' then 'string'
                                            when data_type = 'largetext' then 'string'
                                            when data_type = 'tinyint' then 'boolean'
                                            when data_type = 'bit' then 'boolean'
                                            when data_type = 'datetime' then 'timestamp'
                                            when data_type = 'float' then 'float'
                                            when data_type = 'bigint' then 'bigint'
                                            else 'UNKNOWN'
                                        end as formatted_field_type
                                    from information_schema.columns
                                    where table_schema = 'production'
                                    and table_name not like 'vw_%'
                                    and table_name not like 'staging_%'
                                ) as dat
                                where exists
                                (
									select 1
                                    from information_schema.columns as b
                                    where 
										table_schema = 'production'
                                        and table_name not like 'vw_%'
                                        and table_name not like 'staging_%'
										and dat.table_name = b.table_name
										and b.column_name = 'hash'
                                )
                                order by table_name, column_name
                            ";
                            using (var reader = command.ExecuteReader())
                            {
                                if (reader.HasRows)
                                {
                                    while (reader.Read())
                                    {
                                        var item = new TableGlueFormatModel()
                                        {
                                            TableName = reader.GetString(0),
                                            ColumnName = reader.GetString(1),
                                            DataType = reader.GetString(2),
                                            MaxLength = reader.GetValue(3) == DBNull.Value ? null : (long?)reader.GetInt64(3),
                                            FormattedFieldType = reader.GetString(4),
                                            FormattedString = reader.GetString(5)
                                        };
                                        if (!fullDbOutput.ContainsKey(item.TableName))
                                        {
                                            fullDbOutput.Add(item.TableName, new List<TableGlueFormatModel>());
                                        }
                                        fullDbOutput[item.TableName].Add(item);
                                    }
                                }
                            }
                        }
                    }

                    var droppedTables = originalTableData.Select(x => x.Key).Except(fullDbOutput.Select(x => x.Key)).ToArray();
                    if(droppedTables != null && droppedTables.Length > 0)
                    {
                        foreach(var table in droppedTables)
                        {
                            auditlog.AppendLine($@"{table} dropped");
                        }
                    }

                    if (fullDbOutput.Keys.Count > 0)
                    {
                        newTableConfig.AppendLine(templatePrefix);
                        newTableConfig.AppendLine("table_config = {");

                        foreach (var table in fullDbOutput)
                        {
                            var sensitiveColumns = new List<string>();
                            // Only process tables that have a 'hash' column because it's required.
                            if(!table.Value.Any(x => x.ColumnName == "hash"))
                            {
                                continue;
                            }

                            var tabledata = new TableModel();
                            bool tableAlreadyExists = false;
                            bool requiresVersionBump = false;

                            // For each table, iterate over all the columns to find out whether they are new columns or have changed data types
                            if (originalTableData.ContainsKey(table.Key))
                            {
                                tabledata = originalTableData[table.Key];

                                var originalSchema = JsonConvert.SerializeObject(tabledata.Schema);

                                tabledata.Version = originalTableData[table.Key].Version;
                                tableAlreadyExists = true;

                                try
                                {
                                    var originalPartitions = tabledata.HashPartitions;
                                    using (var conn = DBHelper.GetConnection(activeConnection, true))
                                    {
                                        using (var command = conn.GetCommand())
                                        {
                                            command.CommandText = $"select count(*) from {tabledata.Database}.{table.Key};";
                                            var rowcount = (long)command.ExecuteScalar();
                                            if (rowcount > 0)
                                            {
                                                if(rowcount > 1000000)
                                                {
                                                    tabledata.HashPartitions = 10;
                                                }
                                                else if(rowcount > 100000)
                                                {
                                                    tabledata.HashPartitions = 5;
                                                }
                                                else
                                                {
                                                    tabledata.HashPartitions = 1;
                                                }
                                            }
                                            else
                                            {

                                                tabledata.HashPartitions = 1;
                                            }
                                        }
                                    }
                                    if (originalPartitions != tabledata.HashPartitions)
                                    {
                                        auditlog.AppendLine($@"Updated Hash Partitions for {table.Key} from {originalPartitions} to {tabledata.HashPartitions}");
                                        requiresVersionBump = true;
                                    }

                                    // Dropped columns = Bump version
                                    var droppedColumns = originalTableData[table.Key].Schema.Select(x => x.Name.ToLower()).Except(table.Value.Select(x => x.ColumnName.ToLower())).ToArray();
                                    if (droppedColumns != null && droppedColumns.Length > 0)
                                    {
                                        requiresVersionBump = true;

                                        foreach (var column in droppedColumns)
                                        {
                                            auditlog.AppendLine($@"{table.Key} Dropped Column: {column}");
                                        }
                                    }
                                    // New Column = Bump Version
                                    var newColumns = table.Value.Select(x => x.ColumnName.ToLower()).Except(originalTableData[table.Key].Schema.Select(x => x.Name.ToLower())).ToArray();
                                    if (newColumns != null && newColumns.Length > 0)
                                    {
                                        requiresVersionBump = true;
                                        foreach (var column in newColumns)
                                        {
                                            auditlog.AppendLine($@"{table.Key} New Column: {column}");
                                        }
                                    }
                                }
                                catch (Exception ex)
                                {
                                    Console.WriteLine(ex);
                                    Console.WriteLine("Error processing columns. Aborting.");
                                    Console.ReadLine();
                                    return;
                                }
                                tabledata.Schema.Clear();
                                foreach (var column in table.Value)
                                {
                                    string sensitivity = null;
                                    if (sensitiveColumnNames.ContainsKey(column.ColumnName.ToLower()))
                                    {
                                        sensitivity = sensitiveColumnNames[column.ColumnName.ToLower()];
                                        sensitiveColumns.Add(column.ColumnName.ToLower());
                                    }

                                    tabledata.Schema.Add(new TableSchema()
                                    {
                                        Name = column.ColumnName,
                                        Type = column.FormattedFieldType,
                                        Sensitivity = sensitivity
                                    });

                                    if(originalTableData[table.Key].Schema.Any(x => x.Name == column.ColumnName))
                                    {
                                        var originalColumn = originalTableData[table.Key].Schema.Where(x => x.Name == column.ColumnName).FirstOrDefault();
                                        if (originalColumn.Type != column.FormattedFieldType)
                                        {
                                            auditlog.AppendLine($@"{table.Key} Column {column.ColumnName} Type Changed: {originalColumn.Type} -> {column.FormattedFieldType}");
                                            requiresVersionBump = true;
                                        }
                                    }
                                }

                                // Sensitivity metadata hack, checks if the schema matches on extra metadata points that exist outside of column specific data
                                if (originalSchema != JsonConvert.SerializeObject(tabledata.Schema))
                                {
                                    requiresVersionBump = true;
                                    auditlog.AppendLine("Original schema changed from new table data");
                                }
                            }
                            else
                            {
                                tabledata.Version = 0;
                                tabledata.Schema = new List<TableSchema>();
                                
                                foreach(var column in table.Value)
                                {
                                    string sensitivity = null;
                                    if (sensitiveColumnNames.ContainsKey(column.ColumnName.ToLower()))
                                    {
                                        sensitivity = sensitiveColumnNames[column.ColumnName.ToLower()];
                                        sensitiveColumns.Add(column.ColumnName.ToLower());
                                    }
                                    tabledata.Schema.Add(new TableSchema()
                                    {
                                        Name = column.ColumnName,
                                        Type = column.FormattedFieldType,
                                        Sensitivity = sensitivity
                                    });
                                }


                                auditlog.AppendLine($@"New table: {table.Key}");
                            }

                            // Something we evaluated indicates a version bump is required, and the table previously exists so we need to increment from that value
                            if (requiresVersionBump && tableAlreadyExists)
                            {
                                tabledata.Version += 1;
                            }
                            newTableData.Add(table.Key, tabledata);

                            newTableConfig.AppendLine($"\"{table.Key}\" = <<EOF");
                            newTableConfig.AppendLine(JsonConvert.SerializeObject(tabledata, Formatting.Indented));
                            newTableConfig.AppendLine($@"EOF");
                        }
                        newTableConfig.AppendLine("}");
                        newTableConfig.AppendLine(templatePostfix);
                    }

                    if (auditlog.Length > 0)
                    {
                        if(!newTableData.Any(x => x.Value.Schema == null || x.Value.Schema.Count() == 0))
                        {
                            S3Helper.WriteStringToS3(auditlog.ToString(), "crs-data-export", $"glue-factory/{input.ToLower()}/audit/{DateTime.UtcNow.ToString("yyyy_MM_dd_HH_mm_ss_")}.gz", true);
                            S3Helper.WriteStringToS3(newTableConfig.ToString(), "crs-data-export", $"glue-factory/{input.ToLower()}/data/{DateTime.UtcNow.ToString("yyyy_MM_dd_HH_mm_ss_")}.gz", true);
                            File.Copy($@"{templateOutputPath}{templateOutputFile}", $@"{templateOutputPath}{templateOutputFile}.bak_{DateTime.UtcNow.ToString("yyyy_MM_dd_HH_mm_ss")}");
                            File.WriteAllText($@"{templateOutputPath}{templateOutputFile}", newTableConfig.ToString());
                            var createPath = Path.GetDirectoryName(auditlogWritePath);
                            if (!Directory.Exists(createPath))
                            {
                                Directory.CreateDirectory(createPath);
                            }
                            File.WriteAllText(auditlogWritePath, auditlog.ToString());
                            Console.WriteLine($@"Terraform Updated");
                        }
                        else
                        {
                            Console.WriteLine("Changes aborted due to schema error. A table was detected as blank.");
                            S3Helper.WriteStringToS3(auditlog.ToString(), "crs-data-export", $"glue-factory/{input.ToLower()}/audit/{DateTime.UtcNow.ToString("yyyy_MM_dd_HH_mm_ss_")}.gz", true);
                            File.WriteAllText(auditlogWritePath, "Changes aborted due to schema error. A table was detected as blank.");
                        }
                    }
                    else
                    {
                        Console.WriteLine("No database changes detected.");
                        File.WriteAllText(auditlogWritePath, "No database changes detected.");
                    }
                }
                catch (Exception ex)
                {
                    Console.WriteLine(ex.ToString());
                }
            }
            finally
            {
                Console.WriteLine("Press enter to exit.");
                Console.ReadLine();
            }           
        }
    }
}
