package martian

import (
	"context"
	"fmt"
	"net/http"
	"net/url"
	"time"

	"code.justin.tv/feeds/clients"
	"code.justin.tv/feeds/clients/twitchdoer"
	"code.justin.tv/foundation/twitchclient"
)

const (
	defaultTimingXactName = "cb-martian"
	defaultStatPrefix     = "service.cb-martian."
	defaultStatSampleRate = 1.0
)

// Client interfaces with the Martian service's API.
// This interface satisfies Visage's requirements for a downstream service's client library.
type Client interface {
	GetApplication(ctx context.Context, userID string, reqOpts *twitchclient.ReqOpts) (*ApplicationResponse, error)
	PostApplication(ctx context.Context, params PostApplicationParams, reqOpts *twitchclient.ReqOpts) error
}

// ClientImpl implements the Client interface and uses the twitchhttp client to make http requests
type ClientImpl struct {
	Client twitchclient.Client
}

// NewClient returns a new instance of the Client which uses the given config
func NewClient(conf twitchclient.ClientConf) (Client, error) {
	if conf.TimingXactName == "" {
		conf.TimingXactName = defaultTimingXactName
	}
	twitchClient, err := twitchclient.NewClient(conf)
	if err != nil {
		return nil, err
	}
	return &ClientImpl{Client: twitchClient}, nil
}

func (c *ClientImpl) http(ctx context.Context, statName string, method string, path string, queryParams url.Values, body interface{}, reqOpts *twitchclient.ReqOpts, into interface{}) error {
	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       defaultStatPrefix + statName,
		StatSampleRate: defaultStatSampleRate,
	})

	doer := &twitchdoer.TwitchHTTPDoer{
		Client: c.Client,
		Reqopt: combinedReqOpts,
	}
	return clients.DoHTTP(ctx, doer, method, path, queryParams, body, into, doer.NewTwitchRequest)
}

type ApplicationResponse struct {
	Application Application `json:"application"`
}

type Application struct {
	ResolvedAt *time.Time `json:"resolved_at"`
}

func (c *ClientImpl) GetApplication(ctx context.Context, userID string, reqOpts *twitchclient.ReqOpts) (*ApplicationResponse, error) {
	path := fmt.Sprintf("/v1/users/%s/application", userID)

	var resp ApplicationResponse
	err := c.http(ctx, "get_application", http.MethodGet, path, nil, nil, reqOpts, &resp)
	if err != nil {
		return nil, err
	}

	return &resp, nil
}

// PostApplicationParams contains the parameters for Client.PostApplication().
type PostApplicationParams struct {
	UserID string
	PostApplicationRequestBody
}

// PostApplicationRequestBody is the serialized request body for Client.PostApplication().
type PostApplicationRequestBody struct {
	Category     string `json:"category"`
	CountryCode  string `json:"country_code"`
	Description  string `json:"description"`
	FullName     string `json:"full_name"`
	LanguageCode string `json:"language_code"`
}

// PostApplication makes a POST request to the Martian service to submit an application.
func (c *ClientImpl) PostApplication(ctx context.Context, params PostApplicationParams, reqOpts *twitchclient.ReqOpts) error {
	path := fmt.Sprintf("/v1/users/%s/application", params.UserID)

	return c.http(ctx, "post_application", http.MethodPost, path, nil, params.PostApplicationRequestBody, reqOpts, nil)
}
