// Generate a Salesforce OAuth JWT for development-use.
// The token is used to access the Salesforce API (e.g. query).
// https://help.salesforce.com/articleView?id=remoteaccess_oauth_jwt_flow.htm&type=5

package main

import (
	"flag"
	"fmt"
	"io/ioutil"
	"time"

	"code.justin.tv/cb/martian/internal/jwt"
)

var (
	audience       = flag.String("audience", "", "salesforce authentication url, e.g. https://test.salesforce.com")
	issuer         = flag.String("issuer", "", "salesforce oauth client id")
	privateKeyPath = flag.String("keypath", "", "path to the private key for signing the salesforce oauth jwt")
	subject        = flag.String("subject", "", "username of salesforce user")
)

func init() {
	flag.Parse()
}

func main() {
	privateKey, err := ioutil.ReadFile(*privateKeyPath)
	if err != nil {
		panic(err)
	}

	in5Minutes := time.Now().Add(5 * time.Minute)

	encoded, err := jwt.EncodeRS256(privateKey, jwt.Claims{
		Audience: *audience,
		Expires:  jwt.Expires(in5Minutes),
		Issuer:   *issuer,
		Subject:  *subject,
	})
	if err != nil {
		panic(err)
	}

	fmt.Printf("Encoded JWT (expires at %s):\n", in5Minutes)
	fmt.Println(string(encoded))
}
