package api

import (
	"errors"
	"fmt"
	"net/http"

	"code.justin.tv/cb/martian/client/martian"
	"code.justin.tv/cb/martian/internal/salesforce"
	"code.justin.tv/feeds/service-common"
	"code.justin.tv/web/users-service/client"
	"goji.io/pat"
)

func (s *HTTPServer) getApplication(r *http.Request) (interface{}, error) {
	userID := pat.Param(r, "user_id")

	if userID == "" {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("user id is required"),
		}
	}

	user, err := s.Users.GetUserByID(r.Context(), userID, nil)
	if err != nil {
		switch err.(type) {
		case *client.UserNotFoundError:
			return nil, &service_common.CodedError{
				Code: http.StatusNotFound,
				Err:  fmt.Errorf("user (id: %s) not found", userID),
			}
		default:
			return nil, &service_common.CodedError{
				Code: http.StatusInternalServerError,
				Err:  fmt.Errorf("failed to fetch user (id: %s): %s", userID, err),
			}
		}
	}

	if !userActive(user) {
		return nil, &service_common.CodedError{
			Code: http.StatusUnprocessableEntity,
			Err:  fmt.Errorf("user (id: %s) is deleted, has dmca violation, or has tos violation", userID),
		}
	}

	salesforceCase, err := s.Salesforce.GetCase(r.Context(), userID)
	if err != nil {
		switch err {
		case salesforce.ErrNoCase:
			return nil, &service_common.CodedError{
				Code: http.StatusNotFound,
				Err:  fmt.Errorf("application for user (id: %s) not found", userID),
			}
		default:
			return nil, &service_common.CodedError{
				Code: http.StatusInternalServerError,
				Err:  fmt.Errorf("failed to fetch most recent application/case for user (id: %s): %s", userID, err),
			}
		}
	}

	return martian.ApplicationResponse{
		Application: martian.Application{
			ResolvedAt: salesforceCase.ClosedAt,
		},
	}, nil
}
