package api

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"strconv"
	"sync"
	"time"

	"code.justin.tv/cb/martian/client/martian"
	"code.justin.tv/cb/martian/internal/salesforce"
	"code.justin.tv/cb/martian/internal/twitter"
	"code.justin.tv/cb/martian/internal/youtube"
	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/service-common"
	"code.justin.tv/feeds/spade"
	"code.justin.tv/identity/connections/client"
	"code.justin.tv/revenue/ripley/rpc"
	"code.justin.tv/web/users-service/client"
	"goji.io/pat"
)

func (s *HTTPServer) postApplication(r *http.Request) (interface{}, error) {
	ctx := r.Context()
	userID := pat.Param(r, "user_id")

	if userID == "" {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("user id is required"),
		}
	}

	var reqBody martian.PostApplicationRequestBody
	if err := json.NewDecoder(r.Body).Decode(&reqBody); err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  fmt.Errorf("invalid json request body: %s", err),
		}
	}

	if err := validatePostApplicationRequestBody(reqBody); err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  fmt.Errorf("invalid json request body: %s", err),
		}
	}

	user, err := s.Users.GetUserByID(ctx, userID, nil)
	if err != nil {
		switch err.(type) {
		case *client.UserNotFoundError:
			return nil, &service_common.CodedError{
				Code: http.StatusNotFound,
				Err:  fmt.Errorf("user (id: %s) not found", userID),
			}
		default:
			return nil, &service_common.CodedError{
				Code: http.StatusInternalServerError,
				Err:  fmt.Errorf("failed to fetch user (id: %s): %s", userID, err),
			}
		}
	}

	if !userActive(user) {
		s.Log.LogCtx(ctx, "user_id", userID, log.Msg, "user is deleted, has dmca violation, or has tos violation")

		return nil, &service_common.CodedError{
			Code: http.StatusUnprocessableEntity,
			Err:  fmt.Errorf("user (id: %s) is deleted, has dmca violation, or has tos violation", userID),
		}
	}

	if !userEmailVerified(user) {
		s.Log.LogCtx(ctx, "user_id", userID, log.Msg, "user must have a verified email")

		return nil, &service_common.CodedError{
			Code: http.StatusUnprocessableEntity,
			Err:  fmt.Errorf("user (id: %s) must have a verified email", userID),
		}
	}

	payoutResp, err := s.Ripley.GetPayoutType(ctx, &riptwirp.GetPayoutTypeRequest{
		ChannelId: userID,
	})
	if err != nil {
		s.Log.LogCtx(ctx, log.Err, err, "user_id", userID, log.Msg, "failed to fetch user payout type")

		return nil, &service_common.CodedError{
			Code: http.StatusInternalServerError,
			Err:  fmt.Errorf("failed to fetch user (id: %s) payout type: %s", userID, err),
		}
	}

	if payoutResp.GetPayoutType().GetIsPartner() {
		return nil, &service_common.CodedError{
			Code: http.StatusUnprocessableEntity,
			Err:  fmt.Errorf("user (id: %s) is already partnered and cannot submit an application", userID),
		}
	}

	salesforceCase, err := s.Salesforce.GetCase(ctx, userID)
	if err != nil {
		switch err {
		case salesforce.ErrNoCase: // 👍
		default:
			return nil, &service_common.CodedError{
				Code: http.StatusInternalServerError,
				Err:  fmt.Errorf("failed to create application for user (id: %s): %s", userID, err),
			}
		}
	}

	if salesforceCasePending(salesforceCase) {
		s.Log.LogCtx(ctx, "user_id", userID, log.Msg, "user cannot have more than one pending application")

		return nil, &service_common.CodedError{
			Code: http.StatusUnprocessableEntity,
			Err:  fmt.Errorf("user (id: %s) cannot have more than one pending application", userID),
		}
	}

	if salesforceCaseTooRecent(salesforceCase) {
		s.Log.LogCtx(ctx, "user_id", userID, "salesforce_case_created_at", salesforceCase.CreatedAt, log.Msg, "user may not submit another application within time limit")

		return nil, &service_common.CodedError{
			Code: http.StatusUnprocessableEntity,
			Err:  fmt.Errorf("user (id: %s) may not submit another application within %s since %s", userID, minSalesforceCaseAge, salesforceCase.CreatedAt),
		}
	}

	var wg sync.WaitGroup
	wg.Add(4)

	var broadcastLanguage *string
	go func() {
		broadcastLanguage = s.getBroadcastLanguage(ctx, userID)
		wg.Done()
	}()

	var youTubeID *string
	var youTubeSubscriberCount *int64
	go func() {
		youTubeID, youTubeSubscriberCount = s.getYouTubeData(ctx, userID)
		wg.Done()
	}()

	var twitterID *string
	var twitterFollowerCount *int64
	go func() {
		twitterID, twitterFollowerCount = s.getTwitterData(ctx, userID)
		wg.Done()
	}()

	var affiliateQuestTime, partnerQuestTime *time.Time
	go func() {
		affiliateQuestTime, partnerQuestTime = s.getQuestCompletionTimes(ctx, userID)
		wg.Done()
	}()

	wg.Wait()

	var respBody interface{}
	var salesforceError error

	err = s.Salesforce.CreateCase(ctx, salesforce.CreateCaseParams{
		UserID:                      userID,
		Username:                    *user.Login,
		Email:                       *user.Email,
		FullName:                    reqBody.FullName,
		ContentType:                 reqBody.Category,
		Country:                     reqBody.CountryCode,
		Description:                 reqBody.Description,
		BroadcastLanguage:           broadcastLanguage,
		SuppliedLanguage:            reqBody.LanguageCode,
		ViewerLanguage:              user.Language,
		PathToPartnerCompletionTime: partnerQuestTime,
		TwitterID:                   twitterID,
		TwitterFollowerCount:        twitterFollowerCount,
		YouTubeID:                   youTubeID,
		YouTubeSubscriberCount:      youTubeSubscriberCount,
	})
	if err != nil {
		s.Log.LogCtx(ctx, log.Err, err, "user_id", userID, log.Msg, "failed to create application for user")

		salesforceError = &service_common.CodedError{
			Code: http.StatusInternalServerError,
			Err:  fmt.Errorf("failed to create application for user (id: %s): %s", userID, err),
		}
	} else {
		respBody = martian.ApplicationResponse{
			Application: martian.Application{
				ResolvedAt: salesforceCase.ClosedAt,
			},
		}
	}

	go s.trackToSpade(ctx, trackToSpadeParams{
		PathToAffiliateCompletionTimestamp: affiliateQuestTime,
		PathToPartnerCompletionTimestamp:   partnerQuestTime,
		SalesforceError:                    salesforceError,
		SubmittedBroadcastLanguage:         reqBody.LanguageCode,
		SubmittedCategory:                  reqBody.Category,
		SubmittedCountryCode:               reqBody.CountryCode,
		SubmittedDescription:               reqBody.Description,
		TwitterID:                          twitterID,
		UserID:                             userID,
		YouTubeID:                          youTubeID,
	})

	return respBody, salesforceError
}

func (s *HTTPServer) getBroadcastLanguage(ctx context.Context, userID string) *string {
	channel, err := s.Channels.Get(ctx, userID, nil)
	if err != nil {
		s.Log.LogCtx(ctx, log.Err, err, "user_id", userID, log.Msg, "failed to get channel from channels service")
		return nil
	}

	return &channel.BroadcasterLanguage
}

func (s *HTTPServer) getYouTubeData(ctx context.Context, userID string) (*string, *int64) {
	connection, err := s.Connections.GetYoutubeUser(ctx, userID, nil)
	if err != nil {
		switch err {
		case connections.ErrNoConnection: // 👌
		default:
			s.Log.LogCtx(ctx, log.Err, err, "user_id", userID, log.Msg, "failed to get youtube connection from connections service")
		}
		return nil, nil
	}

	count, err := s.Youtube.GetSubscriberCount(ctx, connection.YoutubeID, youtube.OAuth{
		AccessToken:  connection.Token,
		Expiry:       time.Unix(connection.ExpiresOn, 0),
		RefreshToken: connection.RefreshToken,
	})
	if err != nil {
		s.Log.LogCtx(ctx, log.Err, err, "user_id", userID, log.Msg, "failed to get subscriber count from youtube")
		return nil, nil
	}

	return &connection.YoutubeID, &count
}

func (s *HTTPServer) getTwitterData(ctx context.Context, userID string) (*string, *int64) {
	connection, err := s.Connections.GetTwitterUser(ctx, userID, nil)
	if err != nil {
		switch err {
		case connections.ErrNoConnection: // 👌
		default:
			s.Log.LogCtx(ctx, log.Err, err, "user_id", userID, log.Msg, "failed to get twitter connection from connections service")
		}
		return nil, nil
	}

	id, err := strconv.ParseInt(connection.TwitterID, 10, 64)
	if err != nil {
		s.Log.LogCtx(ctx, log.Err, err, "user_id", userID, "twitter_id", connection.TwitterID, log.Msg, "invalid twitter id")
		return nil, nil
	}

	count, err := s.Twitter.GetFollowerCount(ctx, id, twitter.OAuth{
		AccessToken:  connection.AccessToken,
		AccessSecret: connection.AccessSecret,
	})
	if err != nil {
		s.Log.LogCtx(ctx, log.Err, err, "user_id", userID, log.Msg, "failed to get follower count from twitter")
		return nil, nil
	}

	return &connection.TwitterID, &count
}

func (s *HTTPServer) getQuestCompletionTimes(ctx context.Context, userID string) (*time.Time, *time.Time) {
	resp, err := s.Achievements.GetV1Quests(ctx, userID, nil)
	if err != nil {
		s.Log.LogCtx(ctx, log.Err, err, "user_id", userID, log.Msg, "failed to get quests from achievements service")
		return nil, nil
	}

	var affiliate, partner *time.Time
	for _, quest := range resp.Data.Quests {
		switch quest.Key {
		case "path_to_affiliate":
			affiliate = copyTime(quest.CompletedAt)
		case "path_to_partner":
			partner = copyTime(quest.CompletedAt)
		}
	}

	return affiliate, partner
}

func copyTime(t *time.Time) *time.Time {
	if t == nil {
		return nil
	}

	copy := *t
	return &copy
}

type trackToSpadeParams struct {
	PathToAffiliateCompletionTimestamp *time.Time
	PathToPartnerCompletionTimestamp   *time.Time
	SalesforceError                    error
	SubmittedBroadcastLanguage         string
	SubmittedCategory                  string
	SubmittedCountryCode               string
	SubmittedDescription               string
	TwitterID                          *string
	UserID                             string
	YouTubeID                          *string
}

func (s *HTTPServer) trackToSpade(ctx context.Context, params trackToSpadeParams) {
	userIDInt64, err := strconv.ParseInt(params.UserID, 10, 64)
	if err != nil {
		s.Log.LogCtx(ctx, log.Err, err, "user_id", params.UserID, log.Msg, "cannot parse user id to int64 for spade event")
		return
	}

	var affiliateUnix *float64
	if params.PathToAffiliateCompletionTimestamp != nil {
		unixFloat := float64(params.PathToAffiliateCompletionTimestamp.Unix())
		affiliateUnix = &unixFloat
	}

	var partnerUnix *float64
	if params.PathToPartnerCompletionTimestamp != nil {
		unixFloat := float64(params.PathToPartnerCompletionTimestamp.Unix())
		partnerUnix = &unixFloat
	}

	s.Spade.QueueEvents(spade.Event{
		Name: "server_partnership_application_submission",
		Properties: spadeEventProperties{
			Environment:                        s.Environment,
			PathToAffiliateCompletionTimestamp: affiliateUnix,
			PathToPartnerCompletionTimestamp:   partnerUnix,
			SubmittedBroadcastLanguage:         params.SubmittedBroadcastLanguage,
			SubmittedCategory:                  params.SubmittedCategory,
			SubmittedCountryCode:               params.SubmittedCountryCode,
			SubmittedDescription:               params.SubmittedDescription,
			Success:                            params.SalesforceError == nil,
			TwitterConnected:                   params.TwitterID != nil,
			UserID:                             userIDInt64,
			YouTubeConnected:                   params.YouTubeID != nil,
		},
	})
}

type spadeEventProperties struct {
	Environment                        string   `json:"environment"`
	PathToAffiliateCompletionTimestamp *float64 `json:"path_to_affiliate_completion_timestamp"`
	PathToPartnerCompletionTimestamp   *float64 `json:"path_to_partner_completion_timestamp"`
	SubmittedBroadcastLanguage         string   `json:"submitted_broadcast_language"`
	SubmittedCategory                  string   `json:"submitted_category"`
	SubmittedCountryCode               string   `json:"submitted_country_code"`
	SubmittedDescription               string   `json:"submitted_description"`
	Success                            bool     `json:"success"`
	TwitterConnected                   bool     `json:"twitter_connected"`
	UserID                             int64    `json:"user_id"`
	YouTubeConnected                   bool     `json:"youtube_connected"`
}
