package api

import (
	"context"
	"errors"
	"net/http"
	"strings"
	"testing"
	"time"

	"code.justin.tv/cb/martian/test"
	"goji.io/pattern"
)

func TestPostApplication_FailOnNoUserID(t *testing.T) {
	ctx := context.Background()
	ctx = context.WithValue(ctx, pattern.Variable("user_id"), "")
	server := &HTTPServer{}
	request := &http.Request{}
	request = request.WithContext(ctx)

	result, err := server.postApplication(request)

	if result != nil || err == nil || !strings.Contains(err.Error(), "user id") {
		t.Error("Should fail verifying user id")
	}
}

func TestGetBroadcastLanguage_ChannelsFail(t *testing.T) {
	mock := &test.ChannelsMock{GetError: errors.New("fail")}
	ctx := context.Background()
	server := &HTTPServer{
		Channels: mock,
	}

	result := server.getBroadcastLanguage(ctx, "user_id")

	if result != nil {
		t.Error("Should fail verifying user id")
	}
}

func TestGetYoutubeData_ConnectionsFail(t *testing.T) {
	mock := &test.ConnectionsMock{
		GetYoutubeUserError: errors.New("fail"),
	}
	ctx := context.Background()
	server := &HTTPServer{
		Connections: mock,
	}

	result, number := server.getYouTubeData(ctx, "user_id")

	if number != nil || result != nil {
		t.Error("Should fail getting user")
	}
}

func TestGetTwitterData_ConnectionsFail(t *testing.T) {
	mock := &test.ConnectionsMock{
		GetTwitterUserError: errors.New("fail"),
	}
	ctx := context.Background()
	server := &HTTPServer{
		Connections: mock,
	}

	result, number := server.getTwitterData(ctx, "user_id")

	if number != nil || result != nil {
		t.Error("Should fail getting user")
	}
}

func TestGetQuestCompletionTimes(t *testing.T) {
	expectedAffiliateTime := time.Date(2000, 1, 2, 3, 4, 0, 0, time.UTC)
	expectedPartnerTime := time.Now()

	server := &HTTPServer{
		Achievements: &test.AchievementsMock{
			PathToAffiliateCompletedAt: &expectedAffiliateTime,
			PathToPartnerCompletedAt:   &expectedPartnerTime,
		},
	}

	actualAffiliateTime, actualPartnerTime := server.getQuestCompletionTimes(context.Background(), "user_id")

	if actualAffiliateTime == nil || *actualAffiliateTime != expectedAffiliateTime {
		t.Errorf("expected %s, got %s", expectedAffiliateTime, actualAffiliateTime)
	}

	if actualPartnerTime == nil || *actualPartnerTime != expectedPartnerTime {
		t.Errorf("expected %s, got %s", expectedPartnerTime, actualPartnerTime)
	}
}

func TestTrackToSpade(t *testing.T) {
	mock := test.SpadeMock{}
	expectedAffiliateTime := time.Date(2000, 1, 2, 3, 4, 0, 0, time.UTC)
	expectedPartnerTime := time.Now()
	twitterID := "tweet"
	youTubeID := "maaaawwwwwwwp"

	server := &HTTPServer{
		Spade: &mock,
	}

	server.trackToSpade(context.Background(), trackToSpadeParams{
		UserID:                             "123",
		PathToAffiliateCompletionTimestamp: &expectedAffiliateTime,
		PathToPartnerCompletionTimestamp:   &expectedPartnerTime,
		TwitterID:                          &twitterID,
		YouTubeID:                          &youTubeID,
		SalesforceError:                    nil,
	})

	if len(mock.QueuedEvents) != 1 {
		t.Error("Spade should queue one event")
	}
}
