package api

import (
	"errors"
	"fmt"
	"time"

	"code.justin.tv/cb/martian/client/martian"
	"code.justin.tv/cb/martian/internal/country"
	"code.justin.tv/cb/martian/internal/language"
	"code.justin.tv/cb/martian/internal/salesforce"
	"code.justin.tv/web/users-service/models"
)

const (
	categoryGaming   = "gaming"
	categoryCreative = "creative"

	maxDescriptionLength = 32000
	maxFullNameLength    = 80
)

func validatePostApplicationRequestBody(reqBody martian.PostApplicationRequestBody) error {
	if reqBody.Category != categoryGaming && reqBody.Category != categoryCreative {
		return fmt.Errorf("invalid category, must be one of: %s, %s", categoryGaming, categoryCreative)
	}

	if !country.ValidCode(reqBody.CountryCode) {
		return errors.New("invalid country code")
	}

	if len(reqBody.Description) == 0 || len([]rune(reqBody.Description)) > maxDescriptionLength {
		return fmt.Errorf("description must be between 0 and %d characters", maxDescriptionLength)
	}

	if len(reqBody.FullName) == 0 || len([]rune(reqBody.FullName)) > maxFullNameLength {
		return fmt.Errorf("name must be between 0 and %d characters", maxFullNameLength)
	}

	if !language.ValidCode(reqBody.LanguageCode) {
		return errors.New("invalid language code")
	}

	return nil
}

func userActive(user *models.Properties) bool {
	return user.DeletedOn == nil && !user.HasDmcaViolation() && !user.HasTosViolation()
}

func userEmailVerified(user *models.Properties) bool {
	return user.EmailVerified != nil && *user.EmailVerified
}

func salesforceCasePending(salesforceCase salesforce.Case) bool {
	return salesforceCase.Status == "Open"
}

const (
	day                  = 24 * time.Hour
	minSalesforceCaseAge = 14 * day
)

func salesforceCaseTooRecent(salesforceCase salesforce.Case) bool {
	return time.Now().Add(-minSalesforceCaseAge).Before(salesforceCase.CreatedAt)
}
