package api

import (
	"strings"
	"testing"
	"time"

	"code.justin.tv/cb/martian/client/martian"
	"code.justin.tv/cb/martian/internal/salesforce"
	"code.justin.tv/web/users-service/models"
)

func TestValidationHelpers_InvalidCategory(t *testing.T) {
	reqBody := martian.PostApplicationRequestBody{Category: "wrong"}

	result := validatePostApplicationRequestBody(reqBody)

	if result == nil || !strings.Contains(result.Error(), "invalid category") {
		t.Error("Should return error on invalid category")
	}
}

func TestValidationHelpers_InvalidCountryCode(t *testing.T) {
	reqBody := martian.PostApplicationRequestBody{Category: categoryCreative, CountryCode: "wrong"}

	result := validatePostApplicationRequestBody(reqBody)

	if result == nil || !strings.Contains(result.Error(), "country code") {
		t.Error("Should return error on invalid country code")
	}
}

func TestValidationHelpers_InvalidDescription(t *testing.T) {
	reqBody := martian.PostApplicationRequestBody{Category: categoryCreative, CountryCode: "US", Description: ""}

	result := validatePostApplicationRequestBody(reqBody)

	if result == nil || !strings.Contains(result.Error(), "description") {
		t.Error("Should return error on invalid description")
	}
}

func TestValidationHelpers_InvalidFullname(t *testing.T) {
	reqBody := martian.PostApplicationRequestBody{Category: categoryCreative, CountryCode: "US", Description: "Description", FullName: ""}

	result := validatePostApplicationRequestBody(reqBody)

	if result == nil || !strings.Contains(result.Error(), "name") {
		t.Error("Should return error on invalid name")
	}
}

func TestValidationHelpers_InvalidLanguageCode(t *testing.T) {
	reqBody := martian.PostApplicationRequestBody{Category: categoryCreative, CountryCode: "US", Description: "Description", FullName: "Full Name", LanguageCode: "invalid"}

	result := validatePostApplicationRequestBody(reqBody)

	if result == nil || !strings.Contains(result.Error(), "language") {
		t.Error("Should return error on invalid language code")
	}
}

func TestValidationHelpers_Success(t *testing.T) {
	reqBody := martian.PostApplicationRequestBody{
		Category:     categoryCreative,
		CountryCode:  "US",
		Description:  "Description",
		FullName:     "Full Name",
		LanguageCode: "EN",
	}

	result := validatePostApplicationRequestBody(reqBody)

	if result != nil {
		t.Error("Should return no error")
	}
}

func TestUserActive_FailIfDeleted(t *testing.T) {
	model := &models.Properties{
		DeletedOn: &time.Time{},
	}

	if userActive(model) {
		t.Error("User should be inactive if DeletedOn is filled")
	}
}

func TestUserEmailVerified_FailIfEmailNotVerified(t *testing.T) {
	verified := bool(false)
	model := &models.Properties{
		EmailVerified: &verified,
	}

	if userEmailVerified(model) {
		t.Error("User should fail on Email not verified")
	}
}

func TestSalesforceCasePending_FailOnStatusCheck(t *testing.T) {
	sfCase := salesforce.Case{Status: "invalid"}

	if salesforceCasePending(sfCase) {
		t.Error("SalesForce case check should fail if status is nit Open")
	}
}

func TestSalesforceCaseTooRecent(t *testing.T) {
	tests := []struct {
		Case     salesforce.Case
		Expected bool
		Error    string
	}{
		{
			Case:     salesforce.Case{CreatedAt: time.Now().Add(-minSalesforceCaseAge - time.Minute)},
			Expected: false,
			Error:    "SalesForce case should be just one minute old enough",
		},
		{
			Case:     salesforce.Case{CreatedAt: time.Now()},
			Expected: true,
			Error:    "SalesForce case should be too new (just created)",
		},
		{
			Case:     salesforce.Case{CreatedAt: time.Now().Add(-minSalesforceCaseAge + time.Minute)},
			Expected: true,
			Error:    "SalesForce case should be just one minute too new",
		},
	}

	for _, test := range tests {
		if salesforceCaseTooRecent(test.Case) != test.Expected {
			t.Error(test.Error)
		}
	}
}
