package main

import (
	"net"
	"net/http"
	"os"
	"os/signal"
	"time"

	"code.justin.tv/cb/achievements/client/achievementsinternal"
	"code.justin.tv/cb/martian/cmd/martian/internal/api"
	"code.justin.tv/cb/martian/internal/achievements"
	"code.justin.tv/cb/martian/internal/connections"
	"code.justin.tv/cb/martian/internal/ripley"
	"code.justin.tv/cb/martian/internal/salesforce"
	"code.justin.tv/cb/martian/internal/salesforce/oauth"
	"code.justin.tv/cb/martian/internal/twitter"
	"code.justin.tv/cb/martian/internal/users"
	"code.justin.tv/cb/martian/internal/youtube"
	"code.justin.tv/feeds/service-common"
	"code.justin.tv/feeds/spade"
	"code.justin.tv/feeds/spade/spadedconf"
	"code.justin.tv/foundation/twitchclient"
	identityConnections "code.justin.tv/identity/connections/client"
	"code.justin.tv/revenue/ripley/rpc"
	"code.justin.tv/web/users-service/client/channels"
	"code.justin.tv/web/users-service/client/usersclient_internal"
)

const (
	teamName    = "creator-business"
	serviceName = "martian"
)

// CodeVersion is set by the build script
var CodeVersion string

var instance = service{
	osExit: os.Exit,
	serviceCommon: service_common.ServiceCommon{
		ConfigCommon: service_common.ConfigCommon{
			Team:       teamName,
			Service:    serviceName,
			OsGetenv:   os.Getenv,
			OsHostname: os.Hostname,
		},
		CodeVersion: CodeVersion,
	},
}

type injectables struct {
	Salesforce api.SalesforceClient
	Spade      api.Spade
	Users      api.UsersClient
}

type service struct {
	injectables
	osExit        func(code int)
	onListen      func(net.Addr)
	sigChan       chan os.Signal
	serviceCommon service_common.ServiceCommon
	runner        service_common.ServiceRunner

	configs struct {
		httpConfig            api.HTTPConfig
		connectionsConfig     connections.Config
		ripleyConfig          ripley.Config
		salesforceOAuthConfig oauth.Config
		spadeConfig           spadedconf.Config
		twitterConfig         twitter.Config
		usersConfig           users.Config
		achievementsConfig    achievements.Config
	}

	server api.HTTPServer

	connections  identityConnections.Client
	ripley       riptwirp.Ripley
	twitter      *twitter.Client
	channels     channels.Client
	achievements achievementsinternal.Client
	youtube      *youtube.Client
}

func (s *service) setup() error {
	var err error

	if err = s.serviceCommon.Setup(); err != nil {
		return err
	}

	if err = service_common.LoadConfigs(
		s.serviceCommon.Config,
		&s.configs.httpConfig,
		&s.configs.salesforceOAuthConfig,
		&s.configs.spadeConfig,
		&s.configs.usersConfig,
		&s.configs.ripleyConfig,
		&s.configs.connectionsConfig,
		&s.configs.achievementsConfig,
	); err != nil {
		return err
	}

	if err = s.configs.salesforceOAuthConfig.LoadSecrets(s.serviceCommon.Secrets); err != nil {
		return err
	}

	sandstormDistConf := s.serviceCommon.SetupAdditionalSandstormDistconf("identity", "connections")
	if err = s.configs.twitterConfig.LoadSecrets(sandstormDistConf); err != nil {
		return err
	}

	s.connections, err = identityConnections.NewClient(twitchclient.ClientConf{
		Host: s.configs.connectionsConfig.Host.Get(),
	})
	if err != nil {
		return err
	}

	s.ripley = riptwirp.NewRipleyProtobufClient(
		s.configs.ripleyConfig.Host.Get(),
		twitchclient.NewHTTPClient(twitchclient.ClientConf{
			Host: s.configs.ripleyConfig.Host.Get(),
			Transport: twitchclient.TransportConf{
				MaxIdleConnsPerHost: 1000,
			},
		}),
	)

	if s.Salesforce == nil {
		s.Salesforce = &salesforce.Client{
			Authenticator: &oauth.Client{
				Config: s.configs.salesforceOAuthConfig,
				HTTPClient: &http.Client{
					Timeout: 10 * time.Second,
				},
				Logger: s.serviceCommon.Log,
			},
			HTTPClient: &http.Client{
				Timeout: 10 * time.Second,
			},
			Logger: s.serviceCommon.Log,
		}
	}

	if s.Spade == nil {
		client := spade.Client{
			Config: &s.configs.spadeConfig,
			HTTPClient: &http.Client{
				Timeout: 10 * time.Second,
			},
			Logger: s.serviceCommon.Log,
		}
		client.Setup()
		s.Spade = &client
	}

	s.twitter = &twitter.Client{
		Config: s.configs.twitterConfig,
	}

	if s.Users == nil {
		s.Users, err = usersclient_internal.NewClient(twitchclient.ClientConf{
			Host: s.configs.usersConfig.Host.Get(),
		})
		if err != nil {
			return err
		}
	}

	s.youtube = &youtube.Client{}
	s.achievements, err = achievementsinternal.NewClient(twitchclient.ClientConf{
		Host: s.configs.achievementsConfig.Host.Get(),
	})
	if err != nil {
		return err
	}

	s.channels, err = channels.NewClient(twitchclient.ClientConf{
		Host: s.configs.usersConfig.Host.Get(),
	})
	if err != nil {
		return err
	}

	return nil
}

func (s *service) inject() {
	s.server = api.HTTPServer{
		BaseHTTPServer: service_common.BaseHTTPServer{
			Config: &s.configs.httpConfig.BaseHTTPServerConfig,
			Stats: &service_common.StatSender{
				SubStatter:   s.serviceCommon.Statsd.NewSubStatter("http"),
				ErrorTracker: &s.serviceCommon.ErrorTracker,
			},
			Dims:        &s.serviceCommon.CtxDimensions,
			Log:         s.serviceCommon.Log,
			ElevateKey:  s.serviceCommon.ElevateLogKey,
			Ctxlog:      &s.serviceCommon.Ctxlog,
			OnListen:    s.onListen,
			PanicLogger: s.serviceCommon.PanicLogger,
			XRay:        s.serviceCommon.XRay,
		},
		Config:       &s.configs.httpConfig,
		Environment:  s.serviceCommon.ConfigCommon.Environment,
		Connections:  s.connections,
		Ripley:       s.ripley,
		Salesforce:   s.Salesforce,
		Spade:        s.Spade,
		Twitter:      s.twitter,
		Users:        s.Users,
		Channels:     s.channels,
		Youtube:      s.youtube,
		Achievements: s.achievements,
	}
	s.server.SetupRoutes = s.server.Routes

	s.runner = service_common.ServiceRunner{
		Log: s.serviceCommon.Log,
		Services: []service_common.Service{
			&s.server,
			&s.serviceCommon,
			s.Spade,
		},
		SigChan:      s.sigChan,
		SignalNotify: signal.Notify,
	}
}

func (s *service) main() {
	if err := s.setup(); err != nil {
		if s.serviceCommon.Log != nil {
			s.serviceCommon.Log.Log("err", err, "Unable to load initial config")
		}
		service_common.SetupLogger.Log("err", err, "Unable to load initial config")
		s.osExit(1)
		return
	}
	s.inject()
	if err := s.runner.Execute(); err != nil {
		s.serviceCommon.Log.Log("err", err, "wait to end finished with an error")
		s.osExit(1)
		return
	}
	s.serviceCommon.Log.Log("Finished Main")
}

func main() {
	instance.main()
}
