// +build integration

package main

import (
	"context"
	"fmt"
	"net"
	"net/http"
	"os"
	"strconv"
	"syscall"
	"testing"
	"time"

	"code.justin.tv/cb/martian/client/martian"
	"code.justin.tv/cb/martian/cmd/martian/internal/api/mocks"
	"code.justin.tv/cb/martian/internal/salesforce"
	"code.justin.tv/feeds/clients"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/service-common"
	"code.justin.tv/foundation/twitchclient"
	users_models "code.justin.tv/web/users-service/models"
	"github.com/aws/aws-sdk-go/aws"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

const (
	serverShutdownWaitTime = 3 * time.Second
)

func TestIntegration_GetApplication(t *testing.T) {
	t.Parallel()
	ts := startServer(t, injectables{})
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		client := ts.martianClient
		ctx := ts.ctx
		userID := "244986303" // Test account created for this.  Account is "testmartian"

		Convey("Getting non-existent application should return 404", func() {
			resp, err := client.GetApplication(ctx, userID, nil)
			So(err, ShouldNotBeNil)
			So(clients.ErrorCode(err), ShouldEqual, http.StatusNotFound)
			So(resp, ShouldBeNil)
		})
	})

	ts.onFinish(serverShutdownWaitTime)
}

func TestIntegration_PostApplicationWithMockedSalesforce(t *testing.T) {
	t.Parallel()
	salesforceMock := &mocks.SalesforceClient{}
	ts := startServer(t, injectables{
		Salesforce: salesforceMock,
	})
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	getCreateCaseParams := func(salesforceMock *mocks.SalesforceClient) *salesforce.CreateCaseParams {
		for _, call := range salesforceMock.Mock.Calls {
			if call.Method == "CreateCase" {
				if params, ok := call.Arguments[1].(salesforce.CreateCaseParams); ok {
					return &params
				}
			}
		}
		return nil
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		resetMock(&salesforceMock.Mock)
		client := ts.martianClient
		ctx := ts.ctx
		// Belongs to Ben Hsieh.  We're using staging connections Twitter secret/key.  Ben added his twitter
		// info into the staging connections database so that we can use his user ID for integration testing.
		userID := "136873337"

		Convey("Should create a Salesforce case with YouTube and Twitter data", func() {
			// Initial GetCase should return that there is no case
			salesforceMock.On("GetCase", mock.Anything, userID).Return(salesforce.Case{}, salesforce.ErrNoCase)
			salesforceMock.On("CreateCase", mock.Anything, mock.Anything).Return(nil)

			params := martian.PostApplicationParams{
				PostApplicationRequestBody: martian.PostApplicationRequestBody{
					Category:     "gaming",
					CountryCode:  "US",
					Description:  "test.description",
					FullName:     "test.full_name",
					LanguageCode: "en",
				},
				UserID: userID,
			}

			err := client.PostApplication(ctx, params, nil)
			So(err, ShouldBeNil)

			caseParams := getCreateCaseParams(salesforceMock)
			So(caseParams, ShouldNotBeNil)
			So(caseParams.UserID, ShouldEqual, userID)

			// From the Users service:
			So(caseParams.Username, ShouldEqual, "cupkerk")
			So(caseParams.Email, ShouldEqual, "staging-136873337@example.com")
			So(*caseParams.BroadcastLanguage, ShouldEqual, "")
			So(*caseParams.ViewerLanguage, ShouldEqual, "en")

			// Submitted information:
			So(caseParams.FullName, ShouldEqual, params.FullName)
			So(caseParams.ContentType, ShouldEqual, params.Category)
			So(caseParams.Country, ShouldEqual, params.CountryCode)
			So(caseParams.Description, ShouldEqual, params.Description)
			So(caseParams.SuppliedLanguage, ShouldEqual, params.LanguageCode)

			// From the Twitter connection:
			So(*caseParams.TwitterID, ShouldEqual, "2983337449")
			So(caseParams.TwitterFollowerCount, ShouldNotBeNil)

			// From the YouTube connection:
			So(*caseParams.YouTubeID, ShouldEqual, "UCZUDAmz0g8ALuY1xwLI23tw")
			So(caseParams.YouTubeSubscriberCount, ShouldNotBeNil)

			salesforceMock.AssertExpectations(t)
		})
	})

	ts.onFinish(serverShutdownWaitTime)
}

func TestIntegration_GetPostWithMockedUserService(t *testing.T) {
	t.Parallel()
	usersMock := &mocks.UsersClient{}
	ts := startServer(t, injectables{
		Users: usersMock,
	})
	if ts == nil {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+ts.host, t, func() {
		So(ts.Setup(), ShouldBeNil)
		resetMock(&usersMock.Mock)
		client := ts.martianClient
		ctx := ts.ctx

		Convey("Should get and create a salesforce case", func() {
			userID := timestampUser()
			user := &users_models.Properties{
				EmailVerified: aws.Bool(true),
				Login:         aws.String("test_user"),
				Email:         aws.String("test_user@test.com"),
			}
			usersMock.On("GetUserByID", mock.Anything, userID, mock.Anything).Return(user, nil)

			// Check to make sure that GetApplication returns 404 initially
			a1, e1 := client.GetApplication(ctx, userID, nil)
			So(e1, ShouldNotBeNil)
			So(clients.ErrorCode(e1), ShouldEqual, http.StatusNotFound)
			So(a1, ShouldBeNil)

			// Create the application
			params := martian.PostApplicationParams{
				PostApplicationRequestBody: martian.PostApplicationRequestBody{
					Category:     "gaming",
					CountryCode:  "US",
					Description:  "test.description",
					FullName:     "test.full_name",
					LanguageCode: "en",
				},
				UserID: userID,
			}
			e2 := client.PostApplication(ctx, params, nil)
			So(e2, ShouldBeNil)

			// Make sure that when we fetch the application again we get a response back
			a3, e3 := client.GetApplication(ctx, userID, nil)
			So(e3, ShouldBeNil)
			So(a3, ShouldNotBeNil)
			So(a3.Application.ResolvedAt, ShouldBeNil)

			// Make sure that trying to create another application for the same user fails
			e4 := client.PostApplication(ctx, params, nil)
			So(e4, ShouldNotBeNil)
		})

		Convey("Trying to create an application with should fail ", func() {
			// This works because of a nuance of how convey works.  Nested conveys aren't executed sequentially.
			// They always work from the top level and traverse each nested path.  So something like:
			// Convey("A")
			//  Convey("1")
			//  Convey("2")
			// Actually executes A -> 1, A -> 2.  Not A -> 1, 2
			userID := timestampUser()
			user := &users_models.Properties{
				EmailVerified: aws.Bool(true),
				Login:         aws.String("test_user"),
				Email:         aws.String("test_user@test.com"),
			}
			TestApplication := func() {
				usersMock.On("GetUserByID", mock.Anything, userID, mock.Anything).Return(user, nil)

				// Check to make sure that GetApplication returns 404 initially
				a1, e1 := client.GetApplication(ctx, userID, nil)
				So(e1, ShouldNotBeNil)
				So(clients.ErrorCode(e1), ShouldBeIn, []int{http.StatusNotFound, http.StatusUnprocessableEntity})
				So(a1, ShouldBeNil)

				// Create the application
				params := martian.PostApplicationParams{
					PostApplicationRequestBody: martian.PostApplicationRequestBody{
						Category:     "gaming",
						CountryCode:  "US",
						Description:  "test.description",
						FullName:     "test.full_name",
						LanguageCode: "en",
					},
					UserID: userID,
				}
				e2 := client.PostApplication(ctx, params, nil)
				So(e2, ShouldNotBeNil)
			}

			Convey("Application with unverified user should fail", func() {
				user.EmailVerified = aws.Bool(false)
				TestApplication()
			})

			Convey("Application with deleted user should fail", func() {
				user.DeletedOn = aws.Time(time.Now())
				TestApplication()
			})
			Convey("Application with DMCA'd user should fail", func() {
				user.DmcaViolation = aws.Bool(true)
				TestApplication()
			})
			Convey("Application with TOS violation user should fail", func() {
				user.TermsOfServiceViolation = aws.Bool(true)
				TestApplication()
			})
		})
	})

	ts.onFinish(serverShutdownWaitTime)
}

func resetMock(m *mock.Mock) {
	m.ExpectedCalls = nil
	m.Calls = nil
}

func timestampUser() string {
	return strconv.FormatInt(time.Now().UnixNano(), 10)
}

type testSetup struct {
	ctx           context.Context
	martianClient martian.Client
	httpClient    *http.Client
	host          string
	onFinish      func(timeToWait time.Duration)
	thisInstance  *service
}

func (t *testSetup) Setup() error {
	ctx, cancelFunc := context.WithTimeout(context.Background(), time.Second*5)
	t.ctx = ctx
	Reset(cancelFunc)
	client, err := martian.NewClient(twitchclient.ClientConf{
		Host: t.host,
	})
	if err != nil {
		return err
	}
	t.martianClient = client
	t.httpClient = &http.Client{}
	return nil
}

func addMapValues(m *distconf.InMemory, vals map[string][]byte) error {
	for k, v := range vals {
		if err := m.Write(k, v); err != nil {
			return err
		}
	}
	return nil
}

type panicPanic struct{}

func (p panicPanic) OnPanic(pnc interface{}) {
	panic(pnc)
}

func startServer(t *testing.T, i injectables, configs ...map[string][]byte) *testSetup {
	localConf := &distconf.InMemory{}
	err := addMapValues(localConf, map[string][]byte{
		"martian.listen_addr":  []byte(":0"),
		"rollbar.access_token": []byte(""),
		"statsd.hostport":      []byte(""),
		"debug.addr":           []byte(":0"),
		"logging.to_stdout":    []byte("false"),
		"logging.to_stderr":    []byte("true"),
	})
	if err != nil {
		t.Error(err)
		return nil
	}
	for _, config := range configs {
		err := addMapValues(localConf, config)
		if err != nil {
			t.Error(err)
			return nil
		}
	}

	started := make(chan string)
	finished := make(chan struct{})
	signalToClose := make(chan os.Signal)
	exitCalled := make(chan struct{})
	elevateKey := "hi"
	thisInstance := service{
		injectables: i,
		osExit: func(i int) {
			if i != 0 {
				t.Error("Invalid osExit status code", i)
			}
			close(exitCalled)
		},
		serviceCommon: service_common.ServiceCommon{
			ConfigCommon: service_common.ConfigCommon{
				Team:          teamName,
				Service:       serviceName,
				CustomReaders: []distconf.Reader{localConf},
				BaseDirectory: "../../",
				OsGetenv:      os.Getenv,
				OsHostname:    os.Hostname,
			},
			CodeVersion: CodeVersion,
			Log: &log.ElevatedLog{
				ElevateKey: elevateKey,
				NormalLog: log.ContextLogger{
					Logger: t,
				},
				DebugLog: log.ContextLogger{
					Logger: log.Discard,
				},
				LogToDebug: func(_ ...interface{}) bool {
					return false
				},
			},
			PanicLogger: panicPanic{},
		},
		sigChan: signalToClose,
		onListen: func(listeningAddr net.Addr) {
			started <- fmt.Sprintf("http://localhost:%d", listeningAddr.(*net.TCPAddr).Port)
		},
	}
	thisInstance.serviceCommon.Log.NormalLog.Dims = &thisInstance.serviceCommon.CtxDimensions
	thisInstance.serviceCommon.Log.DebugLog.Dims = &thisInstance.serviceCommon.CtxDimensions
	go func() {
		thisInstance.main()
		close(finished)
	}()

	var addressForIntegrationTests string
	select {
	case <-exitCalled:
		return nil
	case addressForIntegrationTests = <-started:
	case <-time.After(time.Second * 35):
		t.Error("Took to long to start service")
		return nil
	}

	onFinish := func(timeToWait time.Duration) {
		signalToClose <- syscall.SIGTERM
		select {
		case <-finished:
			return
		case <-time.After(timeToWait):
			t.Error("Timed out waiting for server to end")
		}
	}
	return &testSetup{
		host:         addressForIntegrationTests,
		onFinish:     onFinish,
		thisInstance: &thisInstance,
	}
}
