package jwt

import (
	"bytes"
	"encoding/base64"
	"encoding/json"

	"code.justin.tv/common/jwt"
	"github.com/pkg/errors"
)

// EncodeRS256 base64-encodes an RS256 JWT.
func EncodeRS256(privateKey []byte, claims Claims) ([]byte, error) {
	key, err := jwt.ParseRSAPrivateKey(privateKey)
	if err != nil {
		return nil, errors.Wrap(err, "jwt: failed to parse rsa private key")
	}

	rs256 := jwt.RS256(key)

	encoded, err := jwt.Encode(jwt.NewHeader(rs256), claims, rs256)
	if err != nil {
		return nil, errors.Wrap(err, "jwt: failed to encode")
	}

	return encoded, nil
}

// DecodeRS256 decodes a bases64-encoded RS256 JWT.
func DecodeRS256(publicKey []byte, encoded []byte) (Claims, error) {
	key, err := jwt.ParseRSAPublicKey(publicKey)
	if err != nil {
		return Claims{}, errors.Wrap(err, "jwt: failed to parse rsa public key")
	}

	fragment, err := jwt.Parse(encoded)
	if err != nil {
		return Claims{}, errors.Wrap(err, "jwt: failed to parse serialized jwt")
	}

	if err = fragment.Validate(jwt.RSAValidateOnly(jwt.RS256, key)); err != nil {
		return Claims{}, errors.Wrap(err, "jwt: invalid rs256 jwt")
	}

	decoder := json.NewDecoder(base64.NewDecoder(base64.URLEncoding, bytes.NewReader(fragment.Claims())))

	var claims Claims
	if err = decoder.Decode(&claims); err != nil {
		return Claims{}, errors.Wrap(err, "jwt: failed to decode serialized jwt claims")
	}

	return claims, nil
}
