package salesforce

import (
	"strconv"
	"time"
)

// apiTimeFormat is the expected format of Salesforce API's timestamps.
// Use this format for `time.Format()` and `time.Parse()`.
const apiTimeFormat = "2006-01-02T15:04:05.999-0700"

// apiTime eases the JSON serialization and deserialization of timestamp strings
// when interfacing with the Salesforce API.
type apiTime time.Time

// Format returns Time in apiTimeFormat.
func (t apiTime) Format() string {
	return time.Time(t).Format(apiTimeFormat)
}

// MarshalJSON marshals to apiTimeFormat.
func (t apiTime) MarshalJSON() ([]byte, error) {
	return []byte(strconv.Quote(time.Time(t).Format(apiTimeFormat))), nil
}

// UnmarshalJSON unmarshals from apiTimeFormat.
func (t *apiTime) UnmarshalJSON(b []byte) error {
	unquoted, err := strconv.Unquote(string(b))
	if err != nil {
		return err
	}

	parsed, err := time.Parse(apiTimeFormat, unquoted)
	if err != nil {
		return err
	}

	*t = apiTime(parsed)
	return nil
}
