package salesforce

import (
	"context"
	"fmt"
	"io"
	"io/ioutil"
	"net/http"

	"code.justin.tv/cb/martian/internal/salesforce/oauth"
	"code.justin.tv/feeds/log"
	"github.com/pkg/errors"
)

// Authenticator authenticates with Salesforce.
type Authenticator interface {
	Authenticate(ctx context.Context) (oauth.AuthResponse, error)
}

// Client interfaces with the Salesforce API.
// Client is thread-safe for concurrent use.
type Client struct {
	Authenticator Authenticator
	HTTPClient    *http.Client
	Logger        log.Logger
}

// APIVersion is the version of the Salesforce API.
const APIVersion = "v43.0"

func handleResponseError(status int, body io.ReadCloser) error {
	bytes, err := ioutil.ReadAll(body)
	if err != nil {
		return errors.Wrapf(err, "salesforce: failed to decode status %d response body", status)
	}

	return fmt.Errorf("salesforce: unexpected response (status: %d): %s", status, string(bytes))
}
