package salesforce

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"io"
	"net/http"
	"strings"
	"time"

	"code.justin.tv/cb/martian/internal/salesforce/oauth"
	"code.justin.tv/feeds/log"
	"github.com/pkg/errors"
)

// CreateCaseParams is the parameters for Client.CreateCase().
type CreateCaseParams struct {
	UserID      string `json:"TwitchUserID__c"`
	Username    string `json:"TwitchUsername__c"`
	Email       string `json:"SuppliedEmail"`
	FullName    string `json:"SuppliedName"`
	ContentType string `json:"ContentType__c"`
	Country     string `json:"SuppliedCountry__c"`
	Description string `json:"Description"`

	BroadcastLanguage *string `json:"BroadcastLanguage__c"`
	SuppliedLanguage  string  `json:"SuppliedLanguage__c"`
	ViewerLanguage    *string `json:"ViewerLanguage__c"`

	PathToPartnerCompletionTime *time.Time `json:"PathToPartnerCompletionDate__c"`

	TwitterID            *string `json:"TwitterUserID__c"`
	TwitterFollowerCount *int64  `json:"TwitterFollowerCount__c"`

	YouTubeID              *string `json:"YouTubeChannelID__c"`
	YouTubeSubscriberCount *int64  `json:"YouTubeSubscriberCount__c"`
}

// CreateCase makes a POST request to Salesforce.
func (c *Client) CreateCase(ctx context.Context, params CreateCaseParams) error {
	auth, err := c.Authenticator.Authenticate(ctx)
	if err != nil {
		return errors.Wrap(err, "salesforce: failed to authenticate")
	}

	resp, err := c.postCase(ctx, auth, params)
	if err != nil {
		return errors.Wrap(err, "salesforce: failed to create case")
	}

	defer func() {
		if err = resp.Body.Close(); err != nil {
			c.Logger.Log("ctx", ctx, log.Err, err, "salesforce: failed to close response body")
		}
	}()

	switch resp.StatusCode {
	case http.StatusOK, http.StatusCreated:
		return handlePostOK(resp.Body)
	default:
		return handleResponseError(resp.StatusCode, resp.Body)
	}
}

func (c *Client) postCase(ctx context.Context, auth oauth.AuthResponse, params CreateCaseParams) (*http.Response, error) {
	reqURL := fmt.Sprintf("%s/services/data/%s/sobjects/Case", auth.InstanceURL, APIVersion)
	buffer := new(bytes.Buffer)

	if err := json.NewEncoder(buffer).Encode(params); err != nil {
		return nil, err
	}

	req, err := http.NewRequest(http.MethodPost, reqURL, buffer)
	if err != nil {
		return nil, err
	}

	req.Header.Set("Authorization", fmt.Sprintf("%v %v", auth.TokenType, auth.AccessToken))
	req.Header.Set("Content-Type", "application/json")

	return c.HTTPClient.Do(req.WithContext(ctx))
}

type postCaseResponse struct {
	Success bool     `json:"success"`
	ID      string   `json:"id"`
	Errors  []string `json:"errors"`
}

func handlePostOK(body io.ReadCloser) error {
	var resp postCaseResponse

	if err := json.NewDecoder(body).Decode(&resp); err != nil {
		return errors.Wrap(err, "salesforce: failed to decode status 200 response body")
	}

	if resp.Success {
		return nil
	}

	return fmt.Errorf("salesforce: unsuccessful post: %s", strings.Join(resp.Errors, ", "))
}
