package salesforce

import (
	"context"
	"errors"
	"net/http"
	"net/http/httptest"
	"strings"
	"testing"

	"code.justin.tv/cb/martian/internal/salesforce/mocks"
	"code.justin.tv/cb/martian/internal/salesforce/oauth"
	"github.com/stretchr/testify/mock"
)

func TestCreateCase_AuthenticationError(t *testing.T) {
	authenticator := &mocks.Authenticator{}

	authenticator.On("Authenticate", mock.Anything).Return(oauth.AuthResponse{}, errors.New("🔥")).Once()

	client := Client{
		Authenticator: authenticator,
		HTTPClient:    new(http.Client),
	}

	err := client.CreateCase(context.Background(), CreateCaseParams{})
	if err != nil {
		if !strings.Contains(err.Error(), "🔥") {
			t.Errorf("expected %s, got %s", "🔥", err.Error())
		}
	}
}

func TestCreateCase_UnexpectedPostError(t *testing.T) {
	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusTeapot)
		w.Write([]byte("failed!"))
	}))

	defer server.Close()

	authenticator := &mocks.Authenticator{}

	authenticator.On("Authenticate", mock.Anything).Return(oauth.AuthResponse{
		AccessToken: "🍪",
		InstanceURL: server.URL,
	}, nil).Once()

	client := Client{
		Authenticator: authenticator,
		HTTPClient:    server.Client(),
	}

	err := client.CreateCase(context.Background(), CreateCaseParams{})
	if err != nil {
		expected := "salesforce: unexpected response (status: 418): failed!"

		if !strings.Contains(err.Error(), expected) {
			t.Errorf("expected %s, got %s", expected, err.Error())
		}
	}
}

func TestCreateCase_PostOK(t *testing.T) {
	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)

		payload := `
			{
				"id": "1234567890",
				"success": true,
				"errors": []
			}
		`

		_, err := w.Write([]byte(payload))
		if err != nil {
			t.Error(err)
		}
	}))

	defer server.Close()

	authenticator := &mocks.Authenticator{}

	authenticator.On("Authenticate", mock.Anything).Return(oauth.AuthResponse{
		AccessToken: "🍪",
		InstanceURL: server.URL,
	}, nil).Once()

	client := Client{
		Authenticator: authenticator,
		HTTPClient:    server.Client(),
	}

	err := client.CreateCase(context.Background(), CreateCaseParams{})
	if err != nil {
		t.Error(err)
	}
}
