package salesforce

import (
	"context"
	"encoding/json"
	"fmt"
	"io"
	"net/http"
	"net/url"
	"time"

	"code.justin.tv/cb/martian/internal/salesforce/oauth"
	"code.justin.tv/feeds/log"
	"github.com/pkg/errors"
)

// Case belongs to a user
type Case struct {
	CreatedAt time.Time
	ClosedAt  *time.Time
	Status    string
}

// GetCase makes a GET request to query for the most recent case/application of a given userID.
func (c *Client) GetCase(ctx context.Context, userID string) (Case, error) {
	auth, err := c.Authenticator.Authenticate(ctx)
	if err != nil {
		return Case{}, errors.Wrap(err, "salesforce: failed to authenticate")
	}

	resp, err := c.queryCase(ctx, auth, userID)
	if err != nil {
		return Case{}, errors.Wrap(err, "salesforce: failed to query")
	}

	defer func() {
		if err = resp.Body.Close(); err != nil {
			c.Logger.Log("ctx", ctx, log.Err, err, "salesforce: failed to close response body")
		}
	}()

	switch resp.StatusCode {
	case http.StatusOK:
		return handleQueryOK(resp.Body)
	default:
		return Case{}, handleResponseError(resp.StatusCode, resp.Body)
	}
}

func (c *Client) queryCase(ctx context.Context, auth oauth.AuthResponse, userID string) (*http.Response, error) {
	reqURL := fmt.Sprintf("%s/services/data/%s/query", auth.InstanceURL, APIVersion)

	req, err := http.NewRequest(http.MethodGet, reqURL, nil)
	if err != nil {
		return nil, errors.Wrap(err, "salesforce: failed to initialize http request")
	}

	query := fmt.Sprintf(
		`SELECT CreatedDate, ClosedDate, Status
		FROM Case
		WHERE TwitchUserID__c = '%s'
		AND RecordType.DeveloperName = 'Partner_Application'
		ORDER BY CreatedDate DESC LIMIT 1`,
		userID,
	)

	params := url.Values{
		"q": {query},
	}

	req.URL.RawQuery = params.Encode()

	req.Header.Set("Authorization", fmt.Sprintf("%s %s", auth.TokenType, auth.AccessToken))
	req.Header.Set("Content-Type", "application/json")

	return c.HTTPClient.Do(req.WithContext(ctx))
}

type queryCaseResponse struct {
	Done      bool          `json:"done"`
	Records   []queryRecord `json:"records"`
	TotalSize int           `json:"totalSize"`
}

type queryRecord struct {
	Attributes  queryAttributes `json:"attributes"`
	CreatedDate apiTime
	ClosedDate  *apiTime
	Status      string
}

type queryAttributes struct {
	Type string `json:"type"`
	URL  string `json:"url"`
}

func handleQueryOK(body io.ReadCloser) (Case, error) {
	var resp queryCaseResponse

	if err := json.NewDecoder(body).Decode(&resp); err != nil {
		return Case{}, errors.Wrap(err, "salesforce: failed to decode status 200 response body")
	}

	if len(resp.Records) == 0 {
		return Case{}, ErrNoCase
	}

	record := resp.Records[0]

	var closedAt *time.Time
	if record.ClosedDate != nil {
		temp := time.Time(*record.ClosedDate)
		closedAt = &temp
	}

	return Case{
		CreatedAt: time.Time(record.CreatedDate),
		ClosedAt:  closedAt,
		Status:    record.Status,
	}, nil
}
