package salesforce

import (
	"context"
	"errors"
	"net/http"
	"net/http/httptest"
	"strings"
	"testing"

	"code.justin.tv/cb/martian/internal/salesforce/mocks"
	"code.justin.tv/cb/martian/internal/salesforce/oauth"
	"github.com/stretchr/testify/mock"
)

func TestGetCase_AuthenticationError(t *testing.T) {
	authenticator := &mocks.Authenticator{}

	authenticator.On("Authenticate", mock.Anything).Return(oauth.AuthResponse{}, errors.New("🔥")).Once()

	client := Client{
		Authenticator: authenticator,
		HTTPClient:    new(http.Client),
	}

	_, err := client.GetCase(context.Background(), "123")
	if err != nil {
		if !strings.Contains(err.Error(), "🔥") {
			t.Errorf("expected %s, got %s", "🔥", err.Error())
		}
	}
}

func TestGetCase_UnexpectedQueryError(t *testing.T) {
	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusTeapot)
		w.Write([]byte("failed!"))
	}))

	defer server.Close()

	authenticator := &mocks.Authenticator{}

	authenticator.On("Authenticate", mock.Anything).Return(oauth.AuthResponse{
		AccessToken: "🍪",
		InstanceURL: server.URL,
	}, nil).Once()

	client := Client{
		Authenticator: authenticator,
		HTTPClient:    server.Client(),
	}

	_, err := client.GetCase(context.Background(), "123")
	if err != nil {
		expected := "salesforce: unexpected response (status: 418): failed!"

		if !strings.Contains(err.Error(), expected) {
			t.Errorf("expected %s, got %s", expected, err.Error())
		}
	}
}

func TestGetCase_QueryOK(t *testing.T) {
	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)

		payload := `
			{
				"done": true,
				"totalSize": 1,
				"records": [
					{
						"attributes": {
							"type": "Case",
							"url": "http://case.url"
						},
						"CreatedDate": "2018-07-23T21:11:11.000+0000",
						"CloseDate": null,
						"Status": "Open"
					}
				]
			}
		`

		_, err := w.Write([]byte(payload))
		if err != nil {
			t.Error(err)
		}
	}))

	defer server.Close()

	authenticator := &mocks.Authenticator{}

	authenticator.On("Authenticate", mock.Anything).Return(oauth.AuthResponse{
		AccessToken: "🍪",
		InstanceURL: server.URL,
	}, nil).Once()

	client := Client{
		Authenticator: authenticator,
		HTTPClient:    server.Client(),
	}

	recentCase, err := client.GetCase(context.Background(), "123")
	if err != nil {
		t.Error(err)
	}

	if recentCase.CreatedAt.IsZero() {
		t.Error("case 'created at' should not be zero")
	}

	if recentCase.ClosedAt != nil {
		t.Error("case 'closed at' should be nil")
	}

	if recentCase.Status != "Open" {
		t.Errorf("expected %s, got %s", "Open", recentCase.Status)
	}
}
