package oauth

import (
	"context"
	"encoding/json"
	"io/ioutil"
	"net/http"
	"net/url"
	"strings"
	"time"

	"code.justin.tv/cb/martian/internal/jwt"
	"code.justin.tv/feeds/log"
	"github.com/pkg/errors"
)

// Client authenticates with Salesforce.
type Client struct {
	Config     Config
	HTTPClient *http.Client
	Logger     log.Logger
}

// AuthResponse is the expected response body of an OAuth request.
type AuthResponse struct {
	AccessToken string `json:"access_token"`
	ID          string `json:"id"`
	InstanceURL string `json:"instance_url"`
	Scope       string `json:"scope"`
	TokenType   string `json:"token_type"`
}

const (
	jwtExpiration = 5 * time.Minute
	jwtGrantType  = "urn:ietf:params:oauth:grant-type:jwt-bearer"
)

// Authenticate makes a POST request to create a new OAuth token.
//
// Salesforce's JWT OAuth authentication flow:
// https://help.salesforce.com/articleView?id=remoteaccess_oauth_jwt_flow.htm&type=5
func (c *Client) Authenticate(ctx context.Context) (AuthResponse, error) {
	reqURL := c.Config.Host.Get() + "/services/oauth2/token"

	jwt, err := c.generateJWT()
	if err != nil {
		return AuthResponse{}, errors.Wrap(err, "oauth: failed to generate jwt")
	}

	params := url.Values{
		"assertion":  {jwt},
		"grant_type": {jwtGrantType},
	}

	req, err := http.NewRequest(http.MethodPost, reqURL, strings.NewReader(params.Encode()))
	if err != nil {
		return AuthResponse{}, errors.Wrap(err, "oauth: failed to initialize http request")
	}

	req.Header.Add("Content-Type", "application/x-www-form-urlencoded")

	resp, err := c.HTTPClient.Do(req.WithContext(ctx))
	if err != nil {
		return AuthResponse{}, errors.Wrap(err, "oauth: failed make http request")
	}

	defer func() {
		if closeError := resp.Body.Close(); closeError != nil {
			c.Logger.Log("ctx", ctx, "err", closeError, "oauth: failed to close response body")
		}
	}()

	switch resp.StatusCode {
	case http.StatusOK:
		var auth AuthResponse

		if err = json.NewDecoder(resp.Body).Decode(&auth); err != nil {
			return auth, errors.Wrap(err, "oauth: failed to decode status 200 response body")
		}

		return auth, nil
	default:
		body, err := ioutil.ReadAll(resp.Body)
		if err != nil {
			return AuthResponse{}, errors.Wrap(err, "oauth: failed to read response body")
		}

		return AuthResponse{}, errors.Wrapf(err, "oauth: unexpected response: %s (status: %d)", string(body), resp.StatusCode)
	}
}

func (c *Client) generateJWT() (string, error) {
	key := c.Config.RSA256PrivateKey.Get()

	claims := jwt.Claims{
		Audience: c.Config.Host.Get(),
		Expires:  jwt.Expires(time.Now().Add(jwtExpiration)),
		Issuer:   c.Config.ClientID.Get(),
		Subject:  c.Config.Username.Get(),
	}

	encoded, err := jwt.EncodeRS256([]byte(key), claims)
	if err != nil {
		return "", errors.Wrap(err, "oauth: failed to encode rs256 jwt")
	}

	return string(encoded), nil
}
