package oauth

import (
	"context"
	"fmt"
	"io/ioutil"
	"net/http"
	"net/http/httptest"
	"path/filepath"
	"testing"

	"code.justin.tv/cb/martian/internal/jwt"
	"code.justin.tv/feeds/distconf"
)

func TestAuthenticate(t *testing.T) {
	privateKeyPath, err := filepath.Abs("../../jwt/testdata/rsa256.pem")
	if err != nil {
		t.Fatal(err)
	}

	privateKey, err := ioutil.ReadFile(privateKeyPath)
	if err != nil {
		t.Fatal(err)
	}

	publicKeyPath, err := filepath.Abs("../../jwt/testdata/rsa256.pub.pem")
	if err != nil {
		t.Fatal(err)
	}

	publicKey, err := ioutil.ReadFile(publicKeyPath)
	if err != nil {
		t.Fatal(err)
	}

	accessToken := "🍪"
	clientID := "😬"
	instanceURL := "https://instance.url"
	username := "🚀"
	tokenType := "🐻" // "Bearer"

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		var serverError error

		if r.URL.EscapedPath() != "/services/oauth2/token" {
			t.Error("wrong path")
		}
		if serverError = r.ParseForm(); serverError != nil {
			t.Error(serverError)
		}

		claims, serverError := jwt.DecodeRS256(publicKey, []byte(r.Form.Get("assertion")))
		if serverError != nil {
			t.Error(serverError)
		}
		if claims.Issuer != clientID {
			t.Errorf("expected %s, got %s", clientID, claims.Issuer)
		}
		if claims.Subject != username {
			t.Errorf("expected %s, got %s", username, claims.Subject)
		}

		w.WriteHeader(http.StatusOK)

		payload := fmt.Sprintf(`
			{
				"access_token": "%s",
				"id": "ID",
				"instance_url": "%s",
				"scope": "SCOPE",
				"token_type": "%s"
			}
		`, accessToken, instanceURL, tokenType)

		_, err = w.Write([]byte(payload))
		if err != nil {
			t.Error(err)
		}
	}))

	defer server.Close()

	var conf distconf.Distconf

	client := Client{
		Config: Config{
			ClientID:         conf.Str("oauth.client_id", clientID),
			Host:             conf.Str("oauth.host", server.URL),
			RSA256PrivateKey: conf.Str("oauth.rs256_key", string(privateKey)),
			Username:         conf.Str("oauth.username", username),
		},
		HTTPClient: server.Client(),
	}

	resp, err := client.Authenticate(context.Background())
	if err != nil {
		t.Error(err)
	}
	if resp.AccessToken != accessToken {
		t.Errorf("expected %s, got %s", accessToken, resp.AccessToken)
	}
	if resp.InstanceURL != instanceURL {
		t.Errorf("expected %s, got %s", instanceURL, resp.InstanceURL)
	}
	if resp.TokenType != tokenType {
		t.Errorf("expected %s, got %s", tokenType, resp.TokenType)
	}
}
