package youtube

import (
	"time"

	"code.justin.tv/feeds/log"
	"github.com/pkg/errors"
	"golang.org/x/net/context"
	"golang.org/x/oauth2"
	"google.golang.org/api/youtube/v3"
)

// Client struct for Youtube.
type Client struct {
	Logger log.Logger
}

// OAuth is used to authorize requests to the YouTube API.
type OAuth struct {
	AccessToken  string
	Expiry       time.Time
	RefreshToken string
}

// GetSubscriberCount makes a GET request to Youtube apis' V3 using their golang client.
// https://developers.google.com/youtube/v3/docs/
func (c *Client) GetSubscriberCount(ctx context.Context, channelID string, oauth OAuth) (int64, error) {
	tokenSource := oauth2.StaticTokenSource(&oauth2.Token{
		AccessToken:  oauth.AccessToken,
		TokenType:    "Bearer",
		RefreshToken: oauth.RefreshToken,
		Expiry:       oauth.Expiry,
	})

	youtubeClient, err := youtube.New(oauth2.NewClient(ctx, tokenSource))
	if err != nil {
		return 0, errors.Wrap(err, "youtube: failed to create youtube client")
	}

	response, err := youtubeClient.Channels.List("statistics").Id(channelID).Context(ctx).Do()
	if err != nil {
		return 0, errors.Wrapf(err, "youtube: failed to list statistics for channel id %s", channelID)
	}

	// In case of multiple channels, add the subscribers from all channels to get the total subs count.
	var subscriberCount int64
	for _, item := range response.Items {
		subscriberCount += int64(item.Statistics.SubscriberCount)
	}

	return subscriberCount, nil
}
