-- Function for refreshing the timestamp of the `updated_at` column,
-- whenever the record is updated:

CREATE FUNCTION update_updated_at_column()
RETURNS trigger AS $$
BEGIN
  NEW.updated_at = current_timestamp;
  RETURN NEW;
END;
$$ LANGUAGE 'plpgsql';


-- The `events_status` type defines the `status` flavors:
--
-- * 'available': default event status, even if already past
-- * 'admin_deleted': event deleted by an admin user
-- * 'creator_deleted': event deleted by the creator

CREATE TYPE events_status AS ENUM ('available', 'admin_deleted', 'creator_deleted');

-- The `events` table holds onto event records.

CREATE TABLE IF NOT EXISTS events (
  id                    bigserial         PRIMARY KEY,
  creator_id            integer           NOT NULL,
  start_time_utc        timestamp         NOT NULL,
  end_time_utc          timestamp         NOT NULL,
  time_zone_utc_offset  interval          DEFAULT '0' NOT NULL,
  title                 varchar(255)      NOT NULL,
  game_id               integer           NOT NULL,
  description           text,
  status                events_status     DEFAULT 'available' NOT NULL,

  created_at            timestamp         DEFAULT current_timestamp NOT NULL,
  updated_at            timestamp,

  CONSTRAINT events_end_after_start CHECK (end_time_utc > start_time_utc)
);

CREATE INDEX events_creator_id_idx ON events (creator_id);
CREATE INDEX events_start_time_utc_idx ON events (start_time_utc);
CREATE INDEX events_end_time_utc_idx ON events (end_time_utc);
CREATE INDEX events_title_idx ON events (title);
CREATE INDEX events_game_id_idx ON events (game_id);
CREATE INDEX events_status_idx ON events (status);

COMMENT ON COLUMN events.time_zone_utc_offset IS 'Event time zone specified by the event creator, represented as an offset from UTC';

CREATE TRIGGER events_before_update
BEFORE UPDATE ON events
  FOR EACH ROW EXECUTE PROCEDURE update_updated_at_column();
