package main

import (
	"context"
	"database/sql"
	"fmt"
	"strconv"

	log "github.com/Sirupsen/logrus"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	_ "github.com/lib/pq"
)

const (
	awsRegion  = "us-west-2"
	awsProfile = "twitch-feed-dev"
	dbCred     = ""
	dbAddress  = "localhost:5432/oracle?sslmode=disable"
)

// BackfillEventStatsEmailNotifCount is a one-time script for backfilling
// the `event_stats.enabled_user_email_notification_count` column in the
// `oracle` Amazon RDS PostgresQL database.
//
// From the `OracleEventUsers` AWS DynamoDB table, the script sums the number
// of records with email notification enabled for each event, and updates
// the corresponding `event_stats.enabled_user_email_notification_count` field,
// which is defaulted to 0.
func BackfillEventStatsEmailNotifCount() {
	ctx := context.Background()

	// Instantiate DynamoDB client:
	sess := session.Must(session.NewSession())
	config := &aws.Config{
		Credentials: credentials.NewSharedCredentials("", awsProfile),
		Region:      aws.String(awsRegion),
	}

	dynamo := dynamodb.New(sess, config)
	dynamo.AddDebugHandlers()

	// Instantiate RDS SQL client:
	connectionS := fmt.Sprintf("postgres://%s@%s", dbCred, dbAddress)
	db, err := sql.Open("postgres", connectionS)
	if err != nil {
		panic(err)
	}

	err = db.Ping()
	if err != nil {
		panic(err)
	}

	// Instantiate map of event IDs to enabled email notification count:
	eventEnabledEmailNotifCounts := map[int]int{}

	// Scan the DynamoDB `OracleEventUsers` table for all records:
	tableName := aws.String("OracleEventUsers")
	consistentRead := aws.Bool(true)
	projectionExpression := aws.String("EventID")

	var exclusiveStartKey map[string]*dynamodb.AttributeValue
	var eventID int

	for {
		output, err := dynamo.ScanWithContext(ctx, &dynamodb.ScanInput{
			TableName:            tableName,
			ConsistentRead:       consistentRead,
			ProjectionExpression: projectionExpression,
			ExclusiveStartKey:    exclusiveStartKey,
		})

		if err != nil {
			panic(err)
		}

		scannedCount := aws.Int64Value(output.Count)
		log.Println("scanned count:", scannedCount)

		if scannedCount == 0 {
			break
		}

		// Iterate through the records to increment the frequency map:
		for _, item := range output.Items {
			eventID, err = strconv.Atoi(aws.StringValue(item["EventID"].N))
			if err != nil {
				panic(err)
			}

			eventEnabledEmailNotifCounts[eventID]++
		}

		if output.LastEvaluatedKey == nil {
			break
		}

		exclusiveStartKey = output.LastEvaluatedKey
	}

	// Iterate through the frequency map's keys, and update the
	// `event_stats.enabled_user_email_notification_count` field
	// for each event ID:
	query := "UPDATE event_stats SET enabled_user_email_notification_count = $1 WHERE event_id = $2"
	eventsNotUpdated := map[int]int{}

	for eventID, count := range eventEnabledEmailNotifCounts {
		log.Println("event id:", eventID, "email notification enabled count:", count)

		result, err := db.Exec(query, count, eventID)
		if err != nil {
			log.WithFields(log.Fields{
				"event_id": eventID,
				"count":    count,
			}).Error(err)
		}

		rowCount, err := result.RowsAffected()
		if err != nil {
			log.WithFields(log.Fields{
				"event_id": eventID,
				"count":    count,
			}).Error(err)
		}

		if rowCount != 1 {
			eventsNotUpdated[eventID] = count
		}
	}

	for eventID, count := range eventsNotUpdated {
		log.WithFields(log.Fields{
			"event_id": eventID,
			"count":    count,
		}).Warn("event not updated")
	}
}
