package oracle

import (
	"golang.org/x/net/context"

	"code.justin.tv/cb/oracle/view"
	"code.justin.tv/foundation/twitchclient"
)

// Client is the client interface for the Oracle service.
type Client interface {
	// ==== CHANNEL EVENT COVER IMAGE API ====

	// PostV1ChannelEventCoverImage makes a POST request to upload an event
	// cover image for a given channel. The event cover image is pending and
	// NOT publicly accessible until it is associated with an existing event.
	PostV1ChannelEventCoverImage(ctx context.Context, channelID int, input *view.PostV1ChannelEventCoverImageInput, reqOpts *twitchclient.ReqOpts) (*view.PostV1ChannelEventCoverImageOutput, error)

	// ==== EVENTS API ====

	// Deletev1Events makes a DELETE request for an existing event.
	DeleteV1Event(ctx context.Context, id int, reqOpts *twitchclient.ReqOpts) (*view.DeleteV1EventOutput, error)

	// Getv1Events makes a GET request to fetch an existing event.
	GetV1Event(ctx context.Context, id int, reqOpts *twitchclient.ReqOpts) (*view.GetV1EventOutput, error)

	// GetV1AvailableEventList makes a GET request to fetch a list of events
	// based on channel ID.
	GetV1AvailableEventList(ctx context.Context, input *view.GetV1AvailableEventListInput, reqOpts *twitchclient.ReqOpts) (*view.GetV1AvailableEventListOutput, error)

	// PostV1Event makes a POST request to create a new event.
	PostV1Event(ctx context.Context, input *view.PostV1EventInput, reqOpts *twitchclient.ReqOpts) (*view.PostV1EventOutput, error)

	// PutV1Event makes a PUT request to replace an existing event.
	PutV1Event(ctx context.Context, id int, input *view.PutV1EventInput, reqOpts *twitchclient.ReqOpts) (*view.PutV1EventOutput, error)

	// GetV1EventSuggestionsForGame makes a GET request to query for events related to a given game
	GetV1EventSuggestionsForGame(ctx context.Context, gameID int, reqOpts *twitchclient.ReqOpts) (*view.GetV1EventSuggestionsOutput, error)

	// ==== MANAGER EVENTS API ====
	// GetV1AvailableManagerEventList makes a GET request to fetch a list of
	// manager events based on channel ID.
	GetV1AvailableManagerEventList(ctx context.Context, input *view.GetV1AvailableManagerEventListInput, reqOpts *twitchclient.ReqOpts) (*view.GetV1AvailableManagerEventListOutput, error)

	// ==== EVENT NOTIFICATIONS API ====

	// GetV1UserEventNotificationSettings makes a GET request to read a user's
	// notification settings for an existing event.
	GetV1UserEventNotificationSettings(ctx context.Context, userID int, eventID int, requestingUserID int, reqOpts *twitchclient.ReqOpts) (*view.GetV1UserEventNotificationSettingsOutput, error)

	// PutV1UserEventNotificationSettings makes a PUT request to update a
	// user's notification settings for an existing event.
	PutV1UserEventNotificationSettings(ctx context.Context, userID int, eventID int, requestingUserID int, input *view.PutV1UserEventNotificationSettingsInput, reqOpts *twitchclient.ReqOpts) (*view.PutV1UserEventNotificationSettingsOutput, error)

	// GetV1UserEventsNotificationSettings makes a GET request to read a user's
	// notification settings for multiple existing events.
	GetV1UserEventsNotificationSettings(ctx context.Context, userID int, eventID []int, requestingUserID int, reqOpts *twitchclient.ReqOpts) (*view.GetV1UserEventsNotificationSettingsOutput, error)
}

type client struct {
	twitchclient.Client
}

// NewClient instantiates a new client for the Oracle service.
func NewClient(config twitchclient.ClientConf) (Client, error) {
	if config.TimingXactName == "" {
		config.TimingXactName = "cb-oracle"
	}

	twitchClient, err := twitchclient.NewClient(config)
	if err != nil {
		return nil, err
	}

	return &client{twitchClient}, nil
}
