package oracle_test

import (
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"github.com/stretchr/testify/assert"
	"golang.org/x/net/context"

	"code.justin.tv/cb/oracle/client"
	"code.justin.tv/foundation/twitchclient"
)

func TestDeleteV1Event_ServerErrorResponse(t *testing.T) {
	a := assert.New(t)

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusInternalServerError)

		payload := `
			{
				"status": 500,
				"error": "Server Error"
			}
		`

		_, err := w.Write([]byte(payload))
		a.NoError(err)
	}))

	defer server.Close()

	client, err := oracle.NewClient(twitchclient.ClientConf{Host: server.URL})
	a.NoError(err)

	id := 123

	resp, err := client.DeleteV1Event(context.Background(), id, nil)
	a.Error(err)
	a.Nil(resp)
}

func TestDeleteV1Event_JSONDecoderError(t *testing.T) {
	a := assert.New(t)

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)

		payload := `
			{
				"status": 200,
				"message": "malformed JSON string",
			}
		`

		_, err := w.Write([]byte(payload))
		a.NoError(err)
	}))

	defer server.Close()

	client, err := oracle.NewClient(twitchclient.ClientConf{Host: server.URL})
	a.NoError(err)

	id := 123

	resp, err := client.DeleteV1Event(context.Background(), id, nil)
	a.Error(err)
	a.Nil(resp)
}

func TestDeleteV1Event_Success(t *testing.T) {
	a := assert.New(t)

	id := 123
	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		a.Equal(fmt.Sprintf("/v1/events/%d", id), r.URL.EscapedPath())
		a.Equal(http.MethodDelete, r.Method)

		w.WriteHeader(http.StatusOK)

		payload := `
			{
				"status": 200,
				"message": "",
				"meta": {},
				"data": {}
			}
		`

		_, err := w.Write([]byte(payload))
		a.NoError(err)
	}))

	defer server.Close()

	client, err := oracle.NewClient(twitchclient.ClientConf{Host: server.URL})
	a.NoError(err)

	resp, err := client.DeleteV1Event(context.Background(), id, nil)
	a.NoError(err)

	if a.NotNil(resp) {
		a.Equal(http.StatusOK, resp.Status)
		a.Empty(resp.Message)
	}
}
