package oracle_test

import (
	"fmt"
	"net/http"
	"net/http/httptest"
	"strconv"
	"testing"
	"time"

	"github.com/stretchr/testify/assert"
	"golang.org/x/net/context"

	"code.justin.tv/cb/oracle/client"
	"code.justin.tv/cb/oracle/view"
	"code.justin.tv/foundation/twitchclient"
)

func TestGetV1AvailableEventList_MinimalInput(t *testing.T) {
	a := assert.New(t)

	channelID := 123

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		params := r.URL.Query()

		a.Equal("/v1/events", r.URL.EscapedPath())
		a.Equal(params.Get("channel_id"), strconv.Itoa(channelID))
		a.Empty(params.Get("end_time_utc_after"))
		a.Empty(params.Get("end_time_utc_before"))
		a.Empty(params.Get("order_by"))

		w.WriteHeader(http.StatusOK)

		payload := `
			{
				"status": 200,
				"data": []
			}
		`

		_, err := w.Write([]byte(payload))
		a.NoError(err)
	}))

	defer server.Close()

	params := &view.GetV1AvailableEventListInput{
		ChannelID: channelID,
	}

	client, err := oracle.NewClient(twitchclient.ClientConf{Host: server.URL})
	a.NoError(err)

	resp, err := client.GetV1AvailableEventList(context.Background(), params, nil)
	a.NoError(err)

	a.Equal(http.StatusOK, resp.Status)
}

func TestGetV1AvailableEventList_BogusOptionalInput(t *testing.T) {
	a := assert.New(t)

	channelID := 123
	end := time.Now()
	start := end.AddDate(1, 0, 0)

	nonsenseorder := "sure"

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		params := r.URL.Query()

		a.Equal("/v1/events", r.URL.EscapedPath())
		a.Equal(params.Get("channel_id"), strconv.Itoa(channelID))
		a.Equal(params.Get("end_time_utc_after"), end.UTC().Format(time.RFC3339))
		a.Equal(params.Get("end_time_utc_before"), start.UTC().Format(time.RFC3339))
		a.Equal(params.Get("order_by"), nonsenseorder)

		w.WriteHeader(http.StatusOK)

		payload := `
			{
				"status": 200,
				"data": []
			}
		`

		_, err := w.Write([]byte(payload))
		a.NoError(err)
	}))

	defer server.Close()

	// suppose we messed up inputs and they dont make sense -- the query should still succeed
	params := &view.GetV1AvailableEventListInput{
		ChannelID:        channelID,
		EndTimeUTCAfter:  &end,
		EndTimeUTCBefore: &start,
		OrderBy:          &nonsenseorder,
	}

	client, err := oracle.NewClient(twitchclient.ClientConf{Host: server.URL})
	a.NoError(err)

	resp, err := client.GetV1AvailableEventList(context.Background(), params, nil)
	a.NoError(err)

	a.Equal(http.StatusOK, resp.Status)
}

func TestGetV1AvailableEventList_InternalServerError(t *testing.T) {
	a := assert.New(t)

	channelID := 123

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusInternalServerError)

		payload := `
			{
				"status": 500,
				"error": "Some error occurred within the internal api"
			}
		`

		_, err := w.Write([]byte(payload))
		a.NoError(err)
	}))

	defer server.Close()

	params := &view.GetV1AvailableEventListInput{
		ChannelID: channelID,
	}

	client, err := oracle.NewClient(twitchclient.ClientConf{Host: server.URL})
	a.NoError(err)

	_, err = client.GetV1AvailableEventList(context.Background(), params, nil)
	a.Error(err)
	a.Contains(err.Error(), "Some error occurred within the internal api")
}

func TestGetV1AvailableEventList_MalformedResponse(t *testing.T) {
	a := assert.New(t)

	channelID := 123

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)

		payload := `
			{
				"status": 200,
				"message": "malformed JSON string with an extra double-quote""
			}
		`

		_, err := w.Write([]byte(payload))
		a.NoError(err)
	}))

	defer server.Close()

	params := &view.GetV1AvailableEventListInput{
		ChannelID: channelID,
	}

	client, err := oracle.NewClient(twitchclient.ClientConf{Host: server.URL})
	a.NoError(err)

	_, err = client.GetV1AvailableEventList(context.Background(), params, nil)
	a.Error(err)

	a.Contains(err.Error(), "Unable to read response body")
}

func TestGetV1AvailableEventList_Success(t *testing.T) {
	a := assert.New(t)

	channelID := 123
	evAID := 1
	evBID := 2

	hawaii, err := time.LoadLocation("US/Hawaii")
	a.NoError(err)

	timeZoneID := "US/Hawaii"

	now := time.Now()
	evAStartTime := time.Date(
		now.Year(),
		now.Month(),
		now.Day(),
		now.Hour(),
		now.Minute(),
		now.Second(),
		0,
		hawaii,
	)
	evAEndTime := evAStartTime.Add(1 * time.Hour)
	evBStartTime := evAStartTime.Add(8 * time.Hour)
	evBEndTime := evBStartTime.Add(5 * time.Hour)

	evATitle := "A good Title!"
	evBTitle := "I can B a good Title!"

	evBDescription := "Test description!"

	createdAt, err := time.Parse(time.RFC3339, "2017-02-05T01:50:58Z")
	a.NoError(err)

	updatedAt, err := time.Parse(time.RFC3339, "2017-02-05T01:50:58Z")
	a.NoError(err)

	gameID := 9999
	coverImageID := "abcdefghijklmnopqrstuvwxyz1234567890"
	coverImageSourceURL := "http://test.test"
	coverImageURLTemplate := "http://test.test-TEMPLATE"

	evA := view.V1EventView{
		ID:                    evAID,
		ChannelID:             channelID,
		StartTimeUTC:          evAStartTime.UTC(),
		EndTimeUTC:            evAEndTime.UTC(),
		TimeZoneID:            timeZoneID,
		Title:                 evATitle,
		CreatedAtUTC:          createdAt,
		UpdatedAtUTC:          &updatedAt,
		CoverImageID:          &coverImageID,
		CoverImageSourceURL:   &coverImageSourceURL,
		CoverImageURLTemplate: &coverImageURLTemplate,
	}

	evB := view.V1EventView{
		ID:           evBID,
		ChannelID:    channelID,
		StartTimeUTC: evBStartTime.UTC(),
		EndTimeUTC:   evBEndTime.UTC(),
		TimeZoneID:   timeZoneID,
		Title:        evBTitle,
		Description:  &evBDescription,
		CreatedAtUTC: createdAt,
		UpdatedAtUTC: &updatedAt,
		GameID:       gameID,
	}

	order := "asc"

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		params := r.URL.Query()

		a.Equal("/v1/events", r.URL.EscapedPath())
		a.Equal(params.Get("channel_id"), strconv.Itoa(channelID))
		a.Equal(params.Get("end_time_utc_after"), evAStartTime.UTC().Format(time.RFC3339))
		a.Equal(params.Get("end_time_utc_before"), evBEndTime.UTC().Format(time.RFC3339))
		a.Equal(params.Get("order_by"), order)

		w.WriteHeader(http.StatusOK)

		payloadFmt := `
			{
				"status": 200,
				"data": [
					{
						"id": %d,
						"channel_id": %d,
						"start_time_utc": "%s",
						"end_time_utc": "%s",
						"time_zone_id": "%s",
						"title": "%s",
						"created_at_utc": "%s",
						"updated_at_utc": "%s",
						"cover_image_id": "%s",
						"cover_image_source_url": "%s",
						"cover_image_url_template": "%s"
					},
					{
						"id": %d,
						"channel_id": %d,
						"start_time_utc": "%s",
						"end_time_utc": "%s",
						"time_zone_id": "%s",
						"title": "%s",
						"description": "%s",
						"game_id": %d,
						"created_at_utc": "%s",
						"updated_at_utc": "%s"
					}
				]
			}
		`
		payload := fmt.Sprintf(
			payloadFmt,
			evAID,
			channelID,
			evAStartTime.UTC().Format(time.RFC3339),
			evAEndTime.UTC().Format(time.RFC3339),
			timeZoneID,
			evATitle,
			createdAt.Format(time.RFC3339),
			updatedAt.Format(time.RFC3339),
			coverImageID,
			coverImageSourceURL,
			coverImageURLTemplate,

			evBID,
			channelID,
			evBStartTime.UTC().Format(time.RFC3339),
			evBEndTime.UTC().Format(time.RFC3339),
			timeZoneID,
			evBTitle,
			evBDescription,
			gameID,
			createdAt.Format(time.RFC3339),
			updatedAt.Format(time.RFC3339),
		)

		_, err := w.Write([]byte(payload))
		a.NoError(err)
	}))

	defer server.Close()

	params := &view.GetV1AvailableEventListInput{
		ChannelID:        channelID,
		EndTimeUTCAfter:  &evAStartTime,
		EndTimeUTCBefore: &evBEndTime,
		OrderBy:          &order,
	}

	client, err := oracle.NewClient(twitchclient.ClientConf{Host: server.URL})
	a.NoError(err)

	resp, err := client.GetV1AvailableEventList(context.Background(), params, nil)
	a.NoError(err)

	var expecteddata = []*view.V1EventView{
		&evA,
		&evB,
	}

	a.Equal(http.StatusOK, resp.Status)
	if a.NotNil(resp.Data) {
		a.Len(resp.Data, len(expecteddata))

		for i := 0; i < len(resp.Data); i++ {
			a.Equal(resp.Data[i].ID, expecteddata[i].ID)
			a.Equal(resp.Data[i].ChannelID, expecteddata[i].ChannelID)
			a.Equal(resp.Data[i].StartTimeUTC, expecteddata[i].StartTimeUTC)
			a.Equal(resp.Data[i].EndTimeUTC, expecteddata[i].EndTimeUTC)
			a.Equal(resp.Data[i].TimeZoneID, expecteddata[i].TimeZoneID)
			a.Equal(resp.Data[i].Title, expecteddata[i].Title)
			a.Equal(resp.Data[i].Description, expecteddata[i].Description)
			a.Equal(resp.Data[i].CreatedAtUTC, expecteddata[i].CreatedAtUTC)
			a.Equal(resp.Data[i].UpdatedAtUTC, expecteddata[i].UpdatedAtUTC)
			a.Equal(resp.Data[i].GameID, expecteddata[i].GameID)

			if resp.Data[i].CoverImageID != nil && expecteddata[i].CoverImageID != nil {
				a.Equal(*expecteddata[i].CoverImageID, *resp.Data[i].CoverImageID)
			}

			if resp.Data[i].CoverImageSourceURL != nil && expecteddata[i].CoverImageSourceURL != nil {
				a.Equal(*expecteddata[i].CoverImageSourceURL, *resp.Data[i].CoverImageSourceURL)
			}

			if resp.Data[i].CoverImageURLTemplate != nil && expecteddata[i].CoverImageURLTemplate != nil {
				a.Equal(*expecteddata[i].CoverImageURLTemplate, *resp.Data[i].CoverImageURLTemplate)
			}
		}
	}
}
