package oracle

import (
	"net/http"
	"net/url"
	"strconv"
	"time"

	"golang.org/x/net/context"

	"code.justin.tv/cb/oracle/view"
	"code.justin.tv/foundation/twitchclient"
)

// GetV1AvailableManagerEventList makes a batch GET request
// (Content-Type: "application/json") to the Oracle service's v1 API
// to fetch a list of manager events by channelID & time range.
func (c *client) GetV1AvailableManagerEventList(ctx context.Context, input *view.GetV1AvailableManagerEventListInput, reqOpts *twitchclient.ReqOpts) (*view.GetV1AvailableManagerEventListOutput, error) {
	reqURL, err := url.Parse("/v1/manager_events")
	if err != nil {
		return nil, err
	}

	parameters := url.Values{}

	parameters.Add("channel_id", strconv.Itoa(input.ChannelID))

	if input.EndTimeUTCAfter != nil {
		// convert this to UTC just in case
		parameters.Add("end_time_utc_after", input.EndTimeUTCAfter.UTC().Format(time.RFC3339))
	}

	if input.EndTimeUTCBefore != nil {
		// convert this to UTC just in case
		parameters.Add("end_time_utc_before", input.EndTimeUTCBefore.UTC().Format(time.RFC3339))
	}

	if input.OrderBy != nil {
		parameters.Add("order_by", *input.OrderBy)
	}

	reqURL.RawQuery = parameters.Encode()
	path := reqURL.String()

	req, err := c.NewRequest(http.MethodGet, path, nil)
	if err != nil {
		return nil, err
	}

	req.Header.Set("Content-Type", "application/json")

	mergedOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.cb-oracle.get_v1_available_manager_event_list",
		StatSampleRate: 0.1,
	})

	output := &view.GetV1AvailableManagerEventListOutput{}
	_, err = c.DoJSON(ctx, output, req, mergedOpts)
	if err != nil {
		return nil, err
	}

	return output, nil
}
