package oracle_test

import (
	"fmt"
	"net/http"
	"net/http/httptest"
	"strconv"
	"testing"
	"time"

	"github.com/stretchr/testify/assert"
	"golang.org/x/net/context"

	"code.justin.tv/cb/oracle/client"
	"code.justin.tv/cb/oracle/view"
	"code.justin.tv/foundation/twitchclient"
)

func TestGetV1AvailableManagerEventList_MinimalInput(t *testing.T) {
	a := assert.New(t)

	channelID := 123

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		params := r.URL.Query()

		a.Equal("/v1/manager_events", r.URL.EscapedPath())
		a.Equal(params.Get("channel_id"), strconv.Itoa(channelID))
		a.Empty(params.Get("end_time_utc_after"))
		a.Empty(params.Get("end_time_utc_before"))
		a.Empty(params.Get("order_by"))

		w.WriteHeader(http.StatusOK)

		payload := `
			{
				"status": 200,
				"data": []
			}
		`

		_, err := w.Write([]byte(payload))
		a.NoError(err)
	}))

	defer server.Close()

	params := &view.GetV1AvailableManagerEventListInput{
		ChannelID: channelID,
	}

	client, err := oracle.NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	a.NoError(err)

	resp, err := client.GetV1AvailableManagerEventList(context.Background(), params, nil)
	a.NoError(err)

	a.Equal(http.StatusOK, resp.Status)
}

func TestGetV1AvailableManagerEventList_BogusOptionalInput(t *testing.T) {
	a := assert.New(t)

	channelID := 123
	end := time.Now()
	start := end.AddDate(1, 0, 0)

	nonsenseorder := "sure"

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		params := r.URL.Query()

		a.Equal("/v1/manager_events", r.URL.EscapedPath())
		a.Equal(params.Get("channel_id"), strconv.Itoa(channelID))
		a.Equal(params.Get("end_time_utc_after"), end.UTC().Format(time.RFC3339))
		a.Equal(params.Get("end_time_utc_before"), start.UTC().Format(time.RFC3339))
		a.Equal(params.Get("order_by"), nonsenseorder)

		w.WriteHeader(http.StatusOK)

		payload := `
			{
				"status": 200,
				"data": []
			}
		`

		_, err := w.Write([]byte(payload))
		a.NoError(err)
	}))

	defer server.Close()

	// suppose we messed up inputs and they dont make sense -- the query should still succeed
	params := &view.GetV1AvailableManagerEventListInput{
		ChannelID:        channelID,
		EndTimeUTCAfter:  &end,
		EndTimeUTCBefore: &start,
		OrderBy:          &nonsenseorder,
	}

	client, err := oracle.NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	a.NoError(err)

	resp, err := client.GetV1AvailableManagerEventList(context.Background(), params, nil)
	a.NoError(err)

	a.Equal(http.StatusOK, resp.Status)
}

func TestGetV1AvailableManagerEventList_InternalServerError(t *testing.T) {
	a := assert.New(t)

	channelID := 123

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusInternalServerError)

		payload := `
			{
				"status": 500,
				"error": "Some error occurred within the internal api"
			}
		`

		_, err := w.Write([]byte(payload))
		a.NoError(err)
	}))

	defer server.Close()

	params := &view.GetV1AvailableManagerEventListInput{
		ChannelID: channelID,
	}

	client, err := oracle.NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	a.NoError(err)

	_, err = client.GetV1AvailableManagerEventList(context.Background(), params, nil)
	a.Error(err)
	a.Contains(err.Error(), "Some error occurred within the internal api")
}

func TestGetV1AvailableManagerEventList_MalformedResponse(t *testing.T) {
	a := assert.New(t)

	channelID := 123

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)

		payload := `
			{
				"status": 200,
				"message": "malformed JSON string with an extra double-quote""
			}
		`

		_, err := w.Write([]byte(payload))
		a.NoError(err)
	}))

	defer server.Close()

	params := &view.GetV1AvailableManagerEventListInput{
		ChannelID: channelID,
	}

	client, err := oracle.NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	a.NoError(err)

	_, err = client.GetV1AvailableManagerEventList(context.Background(), params, nil)
	a.Error(err)

	a.Contains(err.Error(), "Unable to read response body")
}

func TestGetV1AvailableManagerEventList_Success(t *testing.T) {
	a := assert.New(t)

	channelID := 123
	eventOneID := 1
	eventTwoID := 2

	hawaii, err := time.LoadLocation("US/Hawaii")
	a.NoError(err)

	timeZoneID := "US/Hawaii"

	now := time.Now()
	eventOneStartTime := time.Date(
		now.Year(),
		now.Month(),
		now.Day(),
		now.Hour(),
		now.Minute(),
		now.Second(),
		0,
		hawaii,
	)
	eventOneEndTime := eventOneStartTime.Add(1 * time.Hour)
	eventTwoStartTime := eventOneStartTime.Add(8 * time.Hour)
	eventTwoEndTime := eventTwoStartTime.Add(5 * time.Hour)

	eventOneTitle := "A good Title!"
	eventTwoTitle := "I can B a good Title!"
	eventTwoDescription := "Test description!"

	createdAt, err := time.Parse(time.RFC3339, "2017-02-05T01:50:58Z")
	a.NoError(err)

	updatedAt, err := time.Parse(time.RFC3339, "2017-02-05T01:50:58Z")
	a.NoError(err)

	gameID := 9999
	coverImageID := "abcdefghijklmnopqrstuvwxyz1234567890"
	coverImageSourceURL := "http://test.test"
	coverImageURLTemplate := "http://test.test-TEMPLATE"

	eventOneEmailNotifCount := 666
	eventTwoEmailNotifCount := 5

	eventOne := view.V1ManagerEventView{
		V1EventView: &view.V1EventView{
			ID:                    eventOneID,
			ChannelID:             channelID,
			StartTimeUTC:          eventOneStartTime.UTC(),
			EndTimeUTC:            eventOneEndTime.UTC(),
			TimeZoneID:            timeZoneID,
			Title:                 eventOneTitle,
			CreatedAtUTC:          createdAt,
			UpdatedAtUTC:          &updatedAt,
			CoverImageID:          &coverImageID,
			CoverImageSourceURL:   &coverImageSourceURL,
			CoverImageURLTemplate: &coverImageURLTemplate,
		},
		EmailNotificationCount: eventOneEmailNotifCount,
	}

	eventTwo := view.V1ManagerEventView{
		V1EventView: &view.V1EventView{
			ID:           eventTwoID,
			ChannelID:    channelID,
			StartTimeUTC: eventTwoStartTime.UTC(),
			EndTimeUTC:   eventTwoEndTime.UTC(),
			TimeZoneID:   timeZoneID,
			Title:        eventTwoTitle,
			Description:  &eventTwoDescription,
			CreatedAtUTC: createdAt,
			UpdatedAtUTC: &updatedAt,
			GameID:       gameID,
		},
		EmailNotificationCount: eventTwoEmailNotifCount,
	}

	order := "asc"

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		params := r.URL.Query()

		a.Equal("/v1/manager_events", r.URL.EscapedPath())
		a.Equal(params.Get("channel_id"), strconv.Itoa(channelID))
		a.Equal(params.Get("end_time_utc_after"), eventOneStartTime.UTC().Format(time.RFC3339))
		a.Equal(params.Get("end_time_utc_before"), eventTwoEndTime.UTC().Format(time.RFC3339))
		a.Equal(params.Get("order_by"), order)

		w.WriteHeader(http.StatusOK)

		payloadFmt := `
			{
				"status": 200,
				"data": [
					{
						"id": %d,
						"channel_id": %d,
						"start_time_utc": "%s",
						"end_time_utc": "%s",
						"time_zone_id": "%s",
						"title": "%s",
						"created_at_utc": "%s",
						"updated_at_utc": "%s",
						"cover_image_id": "%s",
						"cover_image_source_url": "%s",
						"cover_image_url_template": "%s",
						"email_notification_count": %d
					},
					{
						"id": %d,
						"channel_id": %d,
						"start_time_utc": "%s",
						"end_time_utc": "%s",
						"time_zone_id": "%s",
						"title": "%s",
						"description": "%s",
						"game_id": %d,
						"created_at_utc": "%s",
						"updated_at_utc": "%s",
						"email_notification_count": %d
					}
				]
			}
		`
		payload := fmt.Sprintf(
			payloadFmt,
			eventOneID,
			channelID,
			eventOneStartTime.UTC().Format(time.RFC3339),
			eventOneEndTime.UTC().Format(time.RFC3339),
			timeZoneID,
			eventOneTitle,
			createdAt.Format(time.RFC3339),
			updatedAt.Format(time.RFC3339),
			coverImageID,
			coverImageSourceURL,
			coverImageURLTemplate,
			eventOneEmailNotifCount,

			eventTwoID,
			channelID,
			eventTwoStartTime.UTC().Format(time.RFC3339),
			eventTwoEndTime.UTC().Format(time.RFC3339),
			timeZoneID,
			eventTwoTitle,
			eventTwoDescription,
			gameID,
			createdAt.Format(time.RFC3339),
			updatedAt.Format(time.RFC3339),
			eventTwoEmailNotifCount,
		)

		_, err := w.Write([]byte(payload))
		a.NoError(err)
	}))

	defer server.Close()

	params := &view.GetV1AvailableManagerEventListInput{
		ChannelID:        channelID,
		EndTimeUTCAfter:  &eventOneStartTime,
		EndTimeUTCBefore: &eventTwoEndTime,
		OrderBy:          &order,
	}

	client, err := oracle.NewClient(twitchclient.ClientConf{
		Host: server.URL,
	})
	a.NoError(err)

	resp, err := client.GetV1AvailableManagerEventList(context.Background(), params, nil)
	a.NoError(err)

	var expectedData = []*view.V1ManagerEventView{
		&eventOne,
		&eventTwo,
	}

	a.Equal(http.StatusOK, resp.Status)
	if a.NotNil(resp.Data) {
		a.Len(resp.Data, len(expectedData))

		for i := 0; i < len(resp.Data); i++ {
			a.Equal(resp.Data[i].ID, expectedData[i].ID)
			a.Equal(resp.Data[i].ChannelID, expectedData[i].ChannelID)
			a.Equal(resp.Data[i].StartTimeUTC, expectedData[i].StartTimeUTC)
			a.Equal(resp.Data[i].EndTimeUTC, expectedData[i].EndTimeUTC)
			a.Equal(resp.Data[i].TimeZoneID, expectedData[i].TimeZoneID)
			a.Equal(resp.Data[i].Title, expectedData[i].Title)
			a.Equal(resp.Data[i].Description, expectedData[i].Description)
			a.Equal(resp.Data[i].CreatedAtUTC, expectedData[i].CreatedAtUTC)
			a.Equal(resp.Data[i].UpdatedAtUTC, expectedData[i].UpdatedAtUTC)
			a.Equal(resp.Data[i].GameID, expectedData[i].GameID)

			if resp.Data[i].CoverImageID != nil && expectedData[i].CoverImageID != nil {
				a.Equal(*expectedData[i].CoverImageID, *resp.Data[i].CoverImageID)
			}

			if resp.Data[i].CoverImageSourceURL != nil && expectedData[i].CoverImageSourceURL != nil {
				a.Equal(*expectedData[i].CoverImageSourceURL, *resp.Data[i].CoverImageSourceURL)
			}

			if resp.Data[i].CoverImageURLTemplate != nil && expectedData[i].CoverImageURLTemplate != nil {
				a.Equal(*expectedData[i].CoverImageURLTemplate, *resp.Data[i].CoverImageURLTemplate)
			}

			a.Equal(resp.Data[i].EmailNotificationCount, expectedData[i].EmailNotificationCount)
		}
	}
}
