package oracle_test

import (
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"
	"time"

	"github.com/stretchr/testify/assert"
	"golang.org/x/net/context"

	"code.justin.tv/cb/oracle/client"
	"code.justin.tv/cb/oracle/view"
	"code.justin.tv/foundation/twitchclient"
)

func TestGetV1Event_Failure(t *testing.T) {
	a := assert.New(t)

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusBadRequest)

		payload := `
			{
				"status": 400,
				"error": "Invalid event ID"
			}
		`

		_, err := w.Write([]byte(payload))
		a.NoError(err)
	}))

	defer server.Close()

	client, err := oracle.NewClient(twitchclient.ClientConf{Host: server.URL})
	a.NoError(err)

	id := 123

	resp, err := client.GetV1Event(context.Background(), id, nil)
	a.Error(err)
	a.Nil(resp)
}

func TestGetV1Event_MalformedResponse(t *testing.T) {
	a := assert.New(t)

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)

		payload := `
			{
				"status": 200,
				"message": "malformed JSON string with an extra double-quote""
			}
		`

		_, err := w.Write([]byte(payload))
		a.NoError(err)
	}))

	defer server.Close()

	client, err := oracle.NewClient(twitchclient.ClientConf{Host: server.URL})
	a.NoError(err)

	id := 123

	resp, err := client.GetV1Event(context.Background(), id, nil)
	a.Error(err)
	a.Nil(resp)
}

func TestGetV1Event_Success(t *testing.T) {
	a := assert.New(t)

	id := 1
	channelID := 123

	hawaii, err := time.LoadLocation("US/Hawaii")
	a.NoError(err)

	now := time.Now()
	hawaiiStartTime := time.Date(
		now.Year(),
		now.Month(),
		now.Day(),
		now.Hour(),
		now.Minute(),
		now.Second(),
		0,
		hawaii,
	)
	hawaiiEndTime := hawaiiStartTime.Add(1 * time.Hour)
	timeZoneID := "Europe/Warsaw"
	title := "Test title!"
	description := "Test description!"

	createdAt, err := time.Parse(time.RFC3339, "2017-02-05T01:50:58Z")
	a.NoError(err)

	coverImageSourceURL := "http://test.test"
	coverImageURLTemplate := "http://test.test-TEMPLATE"
	gameID := 9999

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		a.Equal(fmt.Sprintf("/v1/events/%d", id), r.URL.EscapedPath())
		a.Equal(http.MethodGet, r.Method)

		w.WriteHeader(http.StatusOK)

		payloadFmt := `
			{
				"status": 200,
				"message": "",
				"meta": {
					"status": "available"
				},
				"data": {
					"id": %d,
					"channel_id": %d,
					"start_time_utc": "%s",
					"end_time_utc": "%s",
					"time_zone_id": "%s",
					"title": "%s",
					"description": "%s",
					"game_id": %d,
					"created_at_utc": "%s",
					"cover_image_source_url": "%s",
					"cover_image_url_template": "%s"
				}
			}
		`

		payload := fmt.Sprintf(
			payloadFmt,
			id,
			channelID,
			hawaiiStartTime.UTC().Format(time.RFC3339),
			hawaiiEndTime.UTC().Format(time.RFC3339),
			timeZoneID,
			title,
			description,
			gameID,
			createdAt.Format(time.RFC3339),
			coverImageSourceURL,
			coverImageURLTemplate,
		)

		_, err = w.Write([]byte(payload))
		a.NoError(err)
	}))

	defer server.Close()

	client, err := oracle.NewClient(twitchclient.ClientConf{Host: server.URL})
	a.NoError(err)

	resp, err := client.GetV1Event(context.Background(), id, nil)
	a.NoError(err)

	if a.NotNil(resp) {
		a.Equal(http.StatusOK, resp.Status)
		a.Empty(resp.Message)
		a.Equal(view.EventStatusAvailable, resp.Meta.Status)
		a.Equal(id, resp.Data.ID)
		a.Equal(channelID, resp.Data.ChannelID)
		a.Equal(hawaiiStartTime.UTC(), resp.Data.StartTimeUTC)
		a.Equal(hawaiiEndTime.UTC(), resp.Data.EndTimeUTC)
		a.Equal(timeZoneID, resp.Data.TimeZoneID)
		a.Equal(title, resp.Data.Title)

		if a.NotNil(resp.Data.Description) {
			a.Equal(description, *resp.Data.Description)
		}

		a.Equal(gameID, resp.Data.GameID)
		a.Equal(createdAt, resp.Data.CreatedAtUTC)
		a.Nil(resp.Data.UpdatedAtUTC)

		if a.NotNil(resp.Data.CoverImageSourceURL) {
			a.Equal(coverImageSourceURL, *resp.Data.CoverImageSourceURL)
		}

		if a.NotNil(resp.Data.CoverImageURLTemplate) {
			a.Equal(coverImageURLTemplate, *resp.Data.CoverImageURLTemplate)
		}
	}
}
