package oracle_test

import (
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"github.com/stretchr/testify/assert"
	"golang.org/x/net/context"

	"code.justin.tv/cb/oracle/client"
	"code.justin.tv/foundation/twitchclient"
)

func TestGetV1UserEventNotification_FailureSettings(t *testing.T) {
	a := assert.New(t)

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusBadRequest)

		payload := `
			{
				"status": 400,
				"error": "Invalid user ID"
			}
		`

		_, err := w.Write([]byte(payload))
		a.NoError(err)
	}))

	defer server.Close()

	client, err := oracle.NewClient(twitchclient.ClientConf{Host: server.URL})
	a.NoError(err)

	resp, err := client.GetV1UserEventNotificationSettings(context.Background(), 0, 0, 0, nil)
	a.Error(err)
	a.Nil(resp)
}

func TestGetV1UserEventNotification_MalformedResponseSettings(t *testing.T) {
	a := assert.New(t)

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)

		payload := `
			{
				"status": 200,
				"message": "malformed JSON string with an extra double-quote""
			}
		`

		_, err := w.Write([]byte(payload))
		a.NoError(err)
	}))

	defer server.Close()

	client, err := oracle.NewClient(twitchclient.ClientConf{Host: server.URL})
	a.NoError(err)

	resp, err := client.GetV1UserEventNotificationSettings(context.Background(), 0, 0, 0, nil)
	a.Error(err)
	a.Nil(resp)
}

func TestGetV1UserEventNotification_SuccessSettings(t *testing.T) {
	a := assert.New(t)

	userID := 123
	eventID := 1

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		expectedPath := fmt.Sprintf("/v1/users/%d/notifications/events/%d/settings", userID, eventID)

		a.Equal(expectedPath, r.URL.EscapedPath())
		a.Equal(http.MethodGet, r.Method)

		w.WriteHeader(http.StatusOK)

		payload := `
			{
				"status": 200,
				"message": "",
				"data": {
					"email_enabled": true
				}
			}
		`

		_, err := w.Write([]byte(payload))
		a.NoError(err)
	}))

	defer server.Close()

	client, err := oracle.NewClient(twitchclient.ClientConf{Host: server.URL})
	a.NoError(err)

	resp, err := client.GetV1UserEventNotificationSettings(context.Background(), userID, eventID, userID, nil)
	a.NoError(err)

	if a.NotNil(resp) {
		a.Equal(http.StatusOK, resp.Status)
		a.Empty(resp.Message)
		a.True(resp.Data.EmailEnabled)
	}
}
