package oracle_test

import (
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"github.com/stretchr/testify/assert"
	"golang.org/x/net/context"

	"code.justin.tv/cb/oracle/client"
	"code.justin.tv/cb/oracle/view"
	"code.justin.tv/foundation/twitchclient"
)

func TestPostV1ChannelEventCoverImage_Failure(t *testing.T) {
	a := assert.New(t)

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusBadRequest)

		payload := `
			{
				"status": 400,
				"error": "Invalid channel ID"
			}
		`

		_, err := w.Write([]byte(payload))
		a.NoError(err)
	}))

	defer server.Close()

	client, err := oracle.NewClient(twitchclient.ClientConf{Host: server.URL})
	a.NoError(err)

	channelID := 123
	input := &view.PostV1ChannelEventCoverImageInput{
		Base64EncodedImage: "test",
	}

	resp, err := client.PostV1ChannelEventCoverImage(context.Background(), channelID, input, nil)
	a.Error(err)
	a.Nil(resp)
}

func TestPostV1ChannelEventCoverImage_MalformedResponse(t *testing.T) {
	a := assert.New(t)

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)

		payload := `
			{
				"status": 200,
				"message": "malformed JSON string with an extra double-quote""
			}
		`

		_, err := w.Write([]byte(payload))
		a.NoError(err)
	}))

	defer server.Close()

	client, err := oracle.NewClient(twitchclient.ClientConf{Host: server.URL})
	a.NoError(err)

	channelID := 123
	input := &view.PostV1ChannelEventCoverImageInput{
		Base64EncodedImage: "test",
	}

	resp, err := client.PostV1ChannelEventCoverImage(context.Background(), channelID, input, nil)
	a.Error(err)
	a.Nil(resp)
}

func TestPostV1ChannelEventCoverImage_Success(t *testing.T) {
	a := assert.New(t)

	channelID := 123
	base64EncodedImage := "test"
	imageID := "some_id_99999999"

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		expectedPath := fmt.Sprintf("/v1/channels/%d/event_cover_images", channelID)

		a.Equal(expectedPath, r.URL.EscapedPath())
		a.Equal(http.MethodPost, r.Method)

		reqBody := &view.PostV1ChannelEventCoverImageInput{}
		err := json.NewDecoder(r.Body).Decode(&reqBody)
		a.NoError(err)

		a.Equal(base64EncodedImage, reqBody.Base64EncodedImage)

		w.WriteHeader(http.StatusOK)

		payloadFmt := `
			{
				"status": 200,
				"message": "",
				"data": {
					"id": "%s"
				}
			}
		`

		payload := fmt.Sprintf(payloadFmt, imageID)

		_, err = w.Write([]byte(payload))
		a.NoError(err)
	}))

	defer server.Close()

	client, err := oracle.NewClient(twitchclient.ClientConf{Host: server.URL})
	a.NoError(err)

	input := &view.PostV1ChannelEventCoverImageInput{
		Base64EncodedImage: base64EncodedImage,
	}

	resp, err := client.PostV1ChannelEventCoverImage(context.Background(), channelID, input, nil)
	a.NoError(err)

	if a.NotNil(resp) {
		a.Equal(http.StatusOK, resp.Status)
		a.Empty(resp.Message)
		a.Equal(imageID, resp.Data.ID)
	}
}
