package oracle_test

import (
	"encoding/json"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"
	"time"

	"github.com/stretchr/testify/assert"
	"golang.org/x/net/context"

	"code.justin.tv/cb/oracle/client"
	"code.justin.tv/cb/oracle/view"
	"code.justin.tv/foundation/twitchclient"
)

func TestPostV1Event_Failure(t *testing.T) {
	a := assert.New(t)

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusBadRequest)

		payload := `
			{
				"status": 400,
				"error": "Invalid channel ID"
			}
		`

		_, err := w.Write([]byte(payload))
		a.NoError(err)
	}))

	defer server.Close()

	client, err := oracle.NewClient(twitchclient.ClientConf{Host: server.URL})
	a.NoError(err)

	startTime := time.Now()
	twoYearsLaterTime := startTime.AddDate(2, 0, 0)

	params := &view.PostV1EventInput{
		ChannelID:    123,
		StartTimeUTC: startTime,
		EndTimeUTC:   twoYearsLaterTime,
		TimeZoneID:   "Asia/Bangkok",
		Title:        "Test title!",
	}

	resp, err := client.PostV1Event(context.Background(), params, nil)
	a.Error(err)
	a.Nil(resp)
}

func TestPostV1Event_MalformedResponse(t *testing.T) {
	a := assert.New(t)

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusOK)

		payload := `
			{
				"status": 200,
				"message": "malformed JSON string with an extra double-quote""
			}
		`

		_, err := w.Write([]byte(payload))
		a.NoError(err)
	}))

	defer server.Close()

	client, err := oracle.NewClient(twitchclient.ClientConf{Host: server.URL})
	a.NoError(err)

	startTime := time.Now()
	endTime := startTime.Add(1 * time.Minute)

	params := &view.PostV1EventInput{
		ChannelID:    123,
		StartTimeUTC: startTime,
		EndTimeUTC:   endTime,
		TimeZoneID:   "Egypt",
		Title:        "Test title!",
	}

	resp, err := client.PostV1Event(context.Background(), params, nil)
	a.Error(err)
	a.Nil(resp)
}

func TestPostV1Event_Success(t *testing.T) {
	a := assert.New(t)

	eventID := 1
	channelID := 123

	hawaii, err := time.LoadLocation("US/Hawaii")
	a.NoError(err)

	now := time.Now()
	hawaiiStartTime := time.Date(
		now.Year(),
		now.Month(),
		now.Day(),
		now.Hour(),
		now.Minute(),
		now.Second(),
		0,
		hawaii,
	)
	hawaiiEndTime := hawaiiStartTime.Add(1 * time.Hour)
	timeZoneID := "Europe/Warsaw"
	title := "Test title!"
	description := "Test description!"
	createdAt, err := time.Parse(time.RFC3339, "2017-02-05T01:50:58Z")
	a.NoError(err)

	coverImageID := "someID"
	coverImageSourceURL := "http://test.test"
	coverImageURLTemplate := "http://test.test-TEMPLATE"
	gameID := 9999

	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		a.Equal("/v1/events", r.URL.EscapedPath())
		a.Equal(http.MethodPost, r.Method)

		reqBody := &view.PostV1EventInput{}
		err := json.NewDecoder(r.Body).Decode(&reqBody)
		a.NoError(err)

		a.Equal(channelID, reqBody.ChannelID)
		a.Equal(hawaiiStartTime.UTC(), reqBody.StartTimeUTC)
		a.Equal(hawaiiEndTime.UTC(), reqBody.EndTimeUTC)
		a.Equal(timeZoneID, reqBody.TimeZoneID)
		a.Equal(title, reqBody.Title)
		a.Equal(description, *reqBody.Description)
		a.Equal(gameID, reqBody.GameID)
		a.Equal(coverImageID, *reqBody.CoverImageID)

		w.WriteHeader(http.StatusOK)

		payloadFmt := `
			{
				"status": 200,
				"message": "",
				"meta": {
					"status": "available"
				},
				"data": {
					"id": %d,
					"channel_id": %d,
					"start_time_utc": "%s",
					"end_time_utc": "%s",
					"time_zone_id": "%s",
					"title": "%s",
					"description": "%s",
					"game_id": %d,
					"created_at_utc": "%s",
					"cover_image_source_url": "%s",
					"cover_image_url_template": "%s"
				}
			}
		`

		payload := fmt.Sprintf(
			payloadFmt,
			eventID,
			channelID,
			hawaiiStartTime.UTC().Format(time.RFC3339),
			hawaiiEndTime.UTC().Format(time.RFC3339),
			timeZoneID,
			title,
			description,
			gameID,
			createdAt.Format(time.RFC3339),
			coverImageSourceURL,
			coverImageURLTemplate,
		)

		_, err = w.Write([]byte(payload))
		a.NoError(err)
	}))

	defer server.Close()

	client, err := oracle.NewClient(twitchclient.ClientConf{Host: server.URL})
	a.NoError(err)

	params := &view.PostV1EventInput{
		ChannelID:    channelID,
		StartTimeUTC: hawaiiStartTime,
		EndTimeUTC:   hawaiiEndTime,
		TimeZoneID:   timeZoneID,
		Title:        title,
		Description:  &description,
		GameID:       gameID,
		CoverImageID: &coverImageID,
	}

	resp, err := client.PostV1Event(context.Background(), params, nil)
	a.NoError(err)

	if a.NotNil(resp) {
		a.Equal(http.StatusOK, resp.Status)
		a.Empty(resp.Message)
		a.Equal(view.EventStatusAvailable, resp.Meta.Status)
		a.Equal(eventID, resp.Data.ID)
		a.Equal(channelID, resp.Data.ChannelID)
		a.Equal(hawaiiStartTime.UTC(), resp.Data.StartTimeUTC)
		a.Equal(hawaiiEndTime.UTC(), resp.Data.EndTimeUTC)
		a.Equal(timeZoneID, resp.Data.TimeZoneID)
		a.Equal(title, resp.Data.Title)

		if a.NotNil(resp.Data.Description) {
			a.Equal(description, *resp.Data.Description)
		}

		a.Equal(gameID, resp.Data.GameID)
		a.Equal(createdAt, resp.Data.CreatedAtUTC)

		if a.NotNil(resp.Data.CoverImageSourceURL) {
			a.Equal(coverImageSourceURL, *resp.Data.CoverImageSourceURL)
		}

		if a.NotNil(resp.Data.CoverImageURLTemplate) {
			a.Equal(coverImageURLTemplate, *resp.Data.CoverImageURLTemplate)
		}
	}
}
